use std::collections::HashMap;
use std::time::Instant;

use winit::{
    event::*,
    event_loop::{ControlFlow, EventLoop},
    window::{Window, WindowBuilder},
    dpi::PhysicalSize
};

use crate::{
    Color,
    Scene,
    Resources,
    Vec2,
    RenderData,
    SceneController,
    EnschinContext
};


use wgpu::{
    Instance,
    SurfaceConfiguration,
    DeviceDescriptor,
    Features,
    Limits,
    SurfaceError,
    Backends,
    TextureUsages,
    PresentMode,
    CommandEncoderDescriptor,
    TextureViewDescriptor
};

pub struct Enschin {
    render_data:    Option<RenderData>,
    scenes:         HashMap<&'static str, Scene>,
    resources:      Resources,
    main_scene:     String,
    window:         Window,
    window_size:    Vec2<u32>,
    window_ratio:   f32,
    time:           Instant,
    clear_color:    Color,
    last_time:      f32,
    fps_time:       f32,
    delta_time:     f32,
    total_time:     f32,
    update_time:    f32,
    render_time:    f32,
    fps:            i32,
}

impl Enschin {
    pub fn init<T:'static + SceneController>(game_name: &str, starting_scene: T) {
        let events = EventLoop::new();
        let builder = WindowBuilder::new().with_title(game_name);
        let window = builder.build(&events).unwrap();

        let mut enschin = Enschin::new(window, starting_scene);
        enschin.pre_init_enschin();

        events.run(move |event, _, control_flow| {
            match event {
                Event::Suspended => {
                    if enschin.render_data.is_none() {
                        futures::executor::block_on(enschin.start());
                    }
                },
                Event::Resumed => {
                    if enschin.render_data.is_none() {
                        futures::executor::block_on(enschin.start());
                    }
                },
                Event::WindowEvent {
                    ref event,
                    window_id,
                } if window_id == enschin.window.id() => {
                    enschin.input(event);
                    match event {
                        WindowEvent::CloseRequested => {
                            *control_flow = ControlFlow::Exit
                        },
                        WindowEvent::Resized(physical_size) => {
                            enschin.resize(*physical_size);
                        }
                        WindowEvent::ScaleFactorChanged { new_inner_size, .. } => {
                            enschin.resize(**new_inner_size);
                        }
                        _ => {}
                    }
                },
                Event::RedrawRequested(window_id) if window_id == enschin.window.id() => {
                    if enschin.render_data.is_some() {
                        enschin.update();
                        match enschin.render() {
                            Ok(_) => {}
                            Err(SurfaceError::Lost) => enschin.resize(enschin.window_size.into()),
                            Err(SurfaceError::OutOfMemory) => *control_flow = ControlFlow::Exit,
                            Err(e) => eprintln!("{:?}", e),
                        }
                    }
                }
                Event::MainEventsCleared => {
                    enschin.window.request_redraw();
                }
                _ => {}
            }
        });
    }

    fn new<T:'static + SceneController>(window: Window, starting_scene: T) -> Enschin {
        Self {
            resources:      Resources::new(),
            render_data:    None,
            window_size:    Vec2::new(0, 0),
            window_ratio:   0.0,
            window,
            main_scene:     starting_scene.name().to_string(),
            scenes:         HashMap::from([(starting_scene.name(), Scene::new(Box::new(starting_scene)))]),
            time:           Instant::now(),
            clear_color:    Color::new(0.0, 0.0, 0.0, 1.0),
            last_time:      0.0,
            fps_time:       0.0,
            delta_time:     0.0,
            total_time:     0.0,
            update_time:    0.0,
            render_time:    0.0,
            fps:            0,
        }
    }

    #[cfg(target_os = "android")]
    fn pre_init_enschin(&mut self) {
        
    }

    #[cfg(not(target_os = "android"))]
    fn pre_init_enschin(&mut self) {
        futures::executor::block_on(self.start());
    }

    async fn start(&mut self) {
        self.window_size = self.window.inner_size().into();
        self.window_ratio = self.window_size.x as f32 / self.window_size.y as f32;

        let instance = Instance::new(Backends::all());
        let surface = unsafe { instance.create_surface(&self.window) };
        let adapter = instance.request_adapter(
            &wgpu::RequestAdapterOptions {
                power_preference: wgpu::PowerPreference::default(),
                compatible_surface: Some(&surface),
                force_fallback_adapter: false,
            },
        ).await.unwrap();

        let (device, queue) = adapter.request_device(
            &DeviceDescriptor {
                features: Features::empty(),
                limits: Limits::default(),
                label: None,
            },
            None,
        ).await.unwrap();

            
        let config = SurfaceConfiguration {
            usage: TextureUsages::RENDER_ATTACHMENT,
            format: surface.get_preferred_format(&adapter).unwrap(),
            width: self.window_size.x,
            height: self.window_size.y,
            present_mode: PresentMode::Fifo,
        };
        surface.configure(&device, &config);

        self.resources.create_default(&device, &config);
        let render_data = RenderData::new(&self.resources, device, queue, surface, config);

        let mut context = EnschinContext::new(
            &mut self.window, 
            &mut self.clear_color, 
            &mut self.resources,
            &mut self.main_scene, 
            &render_data, 
            &self.window_size, 
            self.window_ratio,
            self.delta_time, 
            self.total_time, 
            self.update_time, 
            self.render_time, 
            self.fps
        );

        self.scenes.get_mut(context.main_scene.as_str()).unwrap().start(&mut context);
        self.render_data = Some(render_data);
    }


    fn resize(&mut self, new_size: PhysicalSize<u32>) {
        if new_size.width > 0 && new_size.height > 0 {
            let render_data = self.render_data.as_mut().unwrap();
            self.window_size = new_size.into();
            self.window_ratio = self.window_size.x as f32 / self.window_size.y as f32;
            render_data.get_config_mut().width = new_size.width;
            render_data.get_config_mut().height = new_size.height;
            render_data.get_surface().configure(&render_data.get_device(), &render_data.get_config());
            for (_, scene) in self.scenes.iter_mut() {
                scene.resize(self.window_ratio, &render_data);
            }
        }
    }

    fn input(&mut self, event: &WindowEvent) {
        let scene = match self.scenes.get_mut(self.main_scene.as_str()) {
            Some(x) => x,
            None => panic!("Scene {} does not exist!", self.main_scene.as_str())
        };
        scene.input(self.window_size, event);
    }

    fn update(&mut self)  {
        self.update_total_time();
        self.delta_time = self.total_time - self.last_time;
        self.last_time = self.total_time;
        let render_data = self.render_data.as_ref().unwrap();

        let mut enschin_context = EnschinContext::new(
            &mut self.window, 
            &mut self.clear_color, 
            &mut self.resources,
            &mut self.main_scene, 
            render_data, 
            &self.window_size, 
            self.window_ratio,
            self.delta_time, 
            self.total_time, 
            self.update_time, 
            self.render_time, 
            self.fps
        );

        let scene = match self.scenes.get_mut(enschin_context.main_scene.as_str()) {
            Some(x) => x,
            None => panic!("Scene {} does not exist!", enschin_context.main_scene.as_str())
        };
        
        let now = Instant::now();
        scene.update(&mut enschin_context);
        self.update_time = now.elapsed().as_secs_f32();
    }

    pub fn render(&mut self) -> Result<(), SurfaceError> {
        let render_data = self.render_data.as_ref().unwrap();
        
        let output = render_data.get_surface().get_current_texture()?;
        let view = output.texture.create_view(&TextureViewDescriptor::default());

        let mut encoder = render_data.get_device().create_command_encoder(&CommandEncoderDescriptor {
            label: Some("Render Encoder")
        });

        let enschin_context = EnschinContext::new(
            &mut self.window, 
            &mut self.clear_color, 
            &mut self.resources,
            &mut self.main_scene, 
            render_data, 
            &self.window_size, 
            self.window_ratio,
            self.delta_time, 
            self.total_time, 
            self.update_time, 
            self.render_time, 
            self.fps
        );

        {
            let mut render_pass = encoder.begin_render_pass(&wgpu::RenderPassDescriptor {
                label: Some("Render Pass"),
                color_attachments: &[wgpu::RenderPassColorAttachment {
                    view: &view,
                    resolve_target: None,
                    ops: wgpu::Operations {
                        load: wgpu::LoadOp::Clear(enschin_context.clear_color.into_wgpu()),
                        store: true,
                    },
                }],
    
                depth_stencil_attachment: None,
            });
            
            let scene = match self.scenes.get_mut(enschin_context.main_scene.as_str()) {
                Some(x) => x,
                None => panic!("Scene {} does not exist!", enschin_context.main_scene.as_str())
            };    

            let now = Instant::now();
            scene.render(&mut render_pass, &enschin_context);
            self.render_time = now.elapsed().as_secs_f32();
        }
        
        render_data.get_queue().submit(std::iter::once(encoder.finish()));
        output.present();
    

        self.fps += 1;
        if self.total_time > self.fps_time + 1.0 {
            println!("fps: {}\ndelta: {}\n", self.fps, self.delta_time);
            self.fps_time = self.total_time;
            self.fps = 0;
        }

        Ok(())
    }

    fn update_total_time(&mut self) {
        self.total_time = (self.time.elapsed().as_nanos() as f64 / 1000000000.0 as f64) as f32;
    }

}
