use wgpu::{
    Buffer,
    RenderPass,
    util::DeviceExt,
    Device
};

use crate::{
    ComponentData,
    RenderType,
    EnschinContext,
    Renderer,
    Vec2
};

pub struct ComponentType {
    resize: bool,
    is_static: bool,
    always_render: bool,
    instance_buffer: Buffer,
    buffer_size: u32,
    render_type: RenderType,
    pub components: Vec<Box<dyn ComponentData>>,
}

impl ComponentType {

    pub fn new(enschin: &mut EnschinContext, component: Box<dyn ComponentData>) -> Self {
        Self {
            resize: false,
            render_type: component.render_type(),
            is_static: component.is_static(),
            always_render: component.always_render(),
            instance_buffer: enschin.render_data.get_device().create_buffer_init(
                &wgpu::util::BufferInitDescriptor {
                    label: Some("Instance Buffer"),
                    contents: bytemuck::cast_slice(component.get_pos().unwrap().get_matrix().as_ref()),
                    usage: wgpu::BufferUsages::VERTEX | wgpu::BufferUsages::COPY_DST,
                }),
            buffer_size: 0,
            components: vec!(component),
        }
    }

    pub fn add_component(&mut self, new: Box<dyn ComponentData>) {
        self.components.push(new);
        self.resize = true;
    }

    pub fn get_amount(&self) -> usize {
        self.components.len()
    }

    pub fn get_render_type(&self) -> RenderType {
        self.components[0].render_type()
    }

    pub fn buffer(&mut self,  fov: Vec2<f32>, device: &Device) {
        if self.render_type != RenderType::None {
            if self.is_static {
                if self.resize {
                    self.rewrite_buffer(fov, device);
                    self.resize = false;
                }
            } else {
                self.rewrite_buffer(fov, device);
            }
        }
    }

    pub fn render<'a>(&'a mut self, fov: Vec2<f32>, enschin: &'a EnschinContext, render_pass: &mut RenderPass<'a>) {
        self.buffer(fov, enschin.render_data.get_device());
        match self.render_type {
            RenderType::Multi => {
                self.components[0].render(enschin, render_pass);
                render_pass.set_vertex_buffer(1, self.instance_buffer.slice(..));
                render_pass.render_instanced(0..self.buffer_size as _);
            },
            RenderType::Solo => {
                render_pass.set_vertex_buffer(1, self.instance_buffer.slice(..));
                for (index, component) in self.components.iter().enumerate()  {
                    component.render(enschin, render_pass);
                    render_pass.render_instanced(index as u32..(index+1) as u32);
                }
            },
            RenderType::None => {}
        }
    }


    fn rewrite_buffer(&mut self, fov: Vec2<f32>, device: &Device) {

        let instance_data = match self.always_render {
            true => {
                self.components.iter()
                .map(|c| *c.get_pos().unwrap().get_matrix().as_ref()
                ).collect::<Vec<[f32; 16]>>()
            }, false => {
                self.components.iter()
                .filter(|c| !c.get_pos().unwrap().check_offscreen(fov))
                .map(|c| *c.get_pos().unwrap().get_matrix().as_ref()
                ).collect::<Vec<[f32; 16]>>()
            }
        };

        self.buffer_size = instance_data.len() as u32;

        self.instance_buffer = device.create_buffer_init(
            &wgpu::util::BufferInitDescriptor {
                label: Some("Instance Buffer"),
                contents: bytemuck::cast_slice(&instance_data),
                usage: wgpu::BufferUsages::VERTEX | wgpu::BufferUsages::COPY_DST,
            })
    }
}
