use std::ops::*;
use std::mem;
use winit::dpi::PhysicalSize;

#[repr(C)]
#[derive(PartialEq, Eq, Copy, Clone, Debug)]
pub struct Vec2<T> {
    pub x: T,
    pub y: T
}

#[repr(C)]
#[derive(PartialEq, Eq, Copy, Clone, Debug)]
pub struct Vec3<T> {
    pub x: T,
    pub y: T,
    pub z: T
}

#[repr(C)]
#[derive(PartialEq, Eq, Copy, Clone, Debug)]
pub struct Vec4<T> {
    pub x: T,
    pub y: T,
    pub z: T,
    pub w: T
}


macro_rules! impl_vec {
    ($VecN:ident { $($field:ident),+ }, $n:expr) => {

        impl <T>$VecN<T> {
            #[inline]
            pub const fn new($($field: T),+) -> $VecN<T> {
                $VecN { $($field: $field),+ }
            }
        }


        // Arithmetic

        impl <T: Add<Output = T>> Add for $VecN<T> {
            type Output = $VecN<T>;
            fn add(self, v: $VecN<T>) -> $VecN<T> {
                $VecN::new($(self.$field + v.$field),+) 
            }
        }

        impl <T: Sub<Output = T>> Sub for $VecN<T> {
            type Output = $VecN<T>;
            fn sub(self, v: $VecN<T>) -> $VecN<T> {
                $VecN::new($(self.$field - v.$field),+) 
            }
        }

        impl <T: Div<Output = T>> Div for $VecN<T> {
            type Output = $VecN<T>;
            fn div(self, v: $VecN<T>) -> $VecN<T> {
                $VecN::new($(self.$field / v.$field),+) 
            }
        }

        impl <T: Mul<Output = T>> Mul for $VecN<T> {
            type Output = $VecN<T>;
            fn mul(self, v: $VecN<T>) -> $VecN<T> {
                $VecN::new($(self.$field * v.$field),+) 
            }
        }

        impl <T: Rem<Output = T>> Rem for $VecN<T> {
            type Output = $VecN<T>;
            fn rem(self, v: $VecN<T>) -> $VecN<T> {
                $VecN::new($(self.$field % v.$field),+) 
            }
        }

        // Arithmetic assignment

        impl <T: Copy + Clone + AddAssign<T>> AddAssign<$VecN<T>> for $VecN<T> {
            fn add_assign(&mut self, v: $VecN<T>) {
                ($(self.$field += v.$field),+);
            }
        }

        impl <T: Copy + Clone + SubAssign<T>> SubAssign<$VecN<T>> for $VecN<T> {
            fn sub_assign(&mut self, v: $VecN<T>) {
                ($(self.$field -= v.$field),+);
            }
        }

        impl <T: Copy + Clone + MulAssign<T>> MulAssign<$VecN<T>> for $VecN<T> {
            fn mul_assign(&mut self, v: $VecN<T>) {
                ($(self.$field *= v.$field),+);
            }
        }

        impl <T: Copy + Clone + DivAssign<T>> DivAssign<$VecN<T>> for $VecN<T> {
            fn div_assign(&mut self, v: $VecN<T>) {
                ($(self.$field /= v.$field),+);
            }
        }

        impl <T: Copy + Clone + RemAssign<T>> RemAssign<$VecN<T>> for $VecN<T> {
            fn rem_assign(&mut self, v: $VecN<T>) {
                ($(self.$field %= v.$field),+);
            }
        }


        // Index
        impl <T> AsRef<[T; $n]> for $VecN<T> {
            #[inline]
            fn as_ref(&self) -> &[T; $n] {
                unsafe {
                    mem::transmute(self)
                }
            }
        }

        impl <T> AsMut<[T; $n]> for $VecN<T> {
            #[inline]
            fn as_mut(&mut self) -> &mut [T; $n] {
                unsafe { 
                    mem::transmute(self)
                }
            }
        }
        
        impl <T> Index <usize> for $VecN<T> {
            type Output = T;

            #[inline]
            fn index<'a>(&'a self, i: usize) -> &'a T {
                let v: &[T; $n] = self.as_ref();
                &v[i]
            }
        }
        
        impl <T> IndexMut<usize> for $VecN<T> {
            #[inline]
            fn index_mut<'a>(&'a mut self, i: usize) -> &'a mut T{
                let v: &mut [T; $n] = self.as_mut();
                &mut v[i]
            }
        }


        // Other

        impl <S: Neg<Output = S>> Neg for $VecN<S> {
            type Output = $VecN<S>;

            #[inline]
            fn neg(self) -> $VecN<S> { 
                $VecN::new($(-self.$field),+) 
            }
        }


        impl_scaling_ops!($VecN<usize> { $($field),+ });
        impl_scaling_ops!($VecN<u8> { $($field),+ });
        impl_scaling_ops!($VecN<u16> { $($field),+ });
        impl_scaling_ops!($VecN<u32> { $($field),+ });
        impl_scaling_ops!($VecN<u64> { $($field),+ });
        impl_scaling_ops!($VecN<isize> { $($field),+ });
        impl_scaling_ops!($VecN<i8> { $($field),+ });
        impl_scaling_ops!($VecN<i16> { $($field),+ });
        impl_scaling_ops!($VecN<i32> { $($field),+ });
        impl_scaling_ops!($VecN<i64> { $($field),+ });
        impl_scaling_ops!($VecN<f32> { $($field),+ });
        impl_scaling_ops!($VecN<f64> { $($field),+ });
    };
}


macro_rules! impl_scaling_ops {
    ($VecN:ident<$T:ident> { $($field:ident),+ }) => {

        impl Div<$T> for $VecN<$T> {
            type Output = $VecN<$T>;
            fn div(self, v: $T) -> $VecN<$T> {
                $VecN::new($(self.$field / v),+) 
            }
        }

        impl Mul<$T> for $VecN<$T> {
            type Output = $VecN<$T>;
            fn mul(self, v: $T) -> $VecN<$T> {
                $VecN::new($(self.$field * v),+) 
            }
        }

        impl Rem<$T> for $VecN<$T> {
            type Output = $VecN<$T>;
            fn rem(self, v: $T) -> $VecN<$T> {
                $VecN::new($(self.$field % v),+) 
            }
        }

        impl MulAssign<$T> for $VecN<$T> {
            fn mul_assign(&mut self, v: $T) {
                ($(self.$field *= v),+);
            }
        }

        impl DivAssign<$T> for $VecN<$T> {
            fn div_assign(&mut self, v: $T) {
                ($(self.$field /= v),+);
            }
        }

        impl RemAssign<$T> for $VecN<$T> {
            fn rem_assign(&mut self, v: $T) {
                ($(self.$field %= v),+);
            }
        }

    };
}

impl_vec!(Vec2 { x, y }, 2);
impl_vec!(Vec3 { x, y, z }, 3);
impl_vec!(Vec4 { x, y, z, w }, 4);



impl From<Vec2<u32>> for PhysicalSize<u32> {
    fn from(v: Vec2<u32>) -> Self {
        Self {
            width: v.x,
            height: v.y
        }
    }
}


impl From<PhysicalSize<u32>> for Vec2<u32> {
    fn from(v: PhysicalSize<u32>) -> Vec2<u32> {
        Self {
            x: v.width,
            y: v.height
        }
    }
}
