use std::collections::HashMap;
use std::mem;

use winit::{
    event::{
        MouseButton,
        WindowEvent,
        MouseScrollDelta,
        TouchPhase
    } 
};

use crate::{
    Vec2,
    InputEvent,
    InputCode,
    StateAsBool,
    Position
};

pub struct Input {
    cursor_pos: Vec2<f32>,
    absolute_cursor_pos: Vec2<f32>,
    events: HashMap<&'static str, InputEvent>,
    touch: bool,
    last_input: u32,
}

impl Input {
    pub fn new() -> Self {
        Self {
            cursor_pos: Vec2::new(0.0, 0.0),
            absolute_cursor_pos: Vec2::new(0.0, 0.0),
            events: Default::default(),
            touch: false,
            last_input: 170
        }
    }

    pub fn update(&mut self, event: &WindowEvent, camera_pos: Vec2<f32>, window_size: Vec2<u32>, fov: Vec2<f32>) {
        match event {
            WindowEvent::CursorMoved {
                position,
                ..
            } => {
                self.absolute_cursor_pos = Vec2 {
                    x: position.x as f32 / window_size.x as f32 * fov.x * 2.0 - fov.x,
                    y: position.y as f32 / window_size.y as f32 * -fov.y * 2.0 + fov.y
                };
                self.cursor_pos = self.absolute_cursor_pos + camera_pos;
            },
            WindowEvent::Touch(touch) => {
                self.absolute_cursor_pos = Vec2 {
                    x: touch.location.x as f32 / window_size.x as f32 * fov.x * 2.0 - fov.x,
                    y: touch.location.y as f32 / window_size.y as f32 * -fov.y * 2.0 + fov.y
                };
                self.cursor_pos = self.absolute_cursor_pos + camera_pos;
                match touch.phase {
                    TouchPhase::Started | TouchPhase::Moved => {
                        self.touch = true;
                    },
                    TouchPhase::Ended | TouchPhase::Cancelled => {
                        self.touch = false;
                    }
                }
            },
            WindowEvent::KeyboardInput {
                input,
                ..
            } => {
                match input.virtual_keycode {
                    Some(key) => {
                        self.last_input = key as u32;
                        for (_, event) in self.events.iter_mut().filter(|x| x.1.get_code() == self.last_input) {
                            event.update(input.state.as_bool());
                        }
                    },
                    None => {}
                }
            },
            WindowEvent::MouseInput {
                state, 
                button,
                ..
            } => {
                let mouse_button: u32 = match button{
                    MouseButton::Left => 163,
                    MouseButton::Right => 164,
                    MouseButton::Middle => 165,
                    _ => return
                };
                self.last_input = mouse_button;
                for (_, event) in self.events.iter_mut().filter(|x| x.1.get_code() == mouse_button) {
                    event.update(state.as_bool());
                }
            },
            WindowEvent::MouseWheel {
                delta,
                ..
            } => {
                match delta {
                    MouseScrollDelta::LineDelta (x, y) => {
                        if x > &0.0 {
                            for (_, event) in self.events.iter_mut().filter(|x| x.1.code == InputCode::ScrollRight) {
                                event.update(true);
                            }
                        } else {
                            for (_, event) in self.events.iter_mut().filter(|x| x.1.code == InputCode::ScrollLeft) {
                                event.update(true);
                            }
                        }

                        if y > &0.0 {
                            for (_, event) in self.events.iter_mut().filter(|x| x.1.code == InputCode::ScrollUp) {
                                event.update(true);
                            }
                        } else {
                            for (_, event) in self.events.iter_mut().filter(|x| x.1.code == InputCode::ScrollDown) {
                                event.update(true);
                            }
                        }
                    },
                    _ => {}
                }
            },
            _ => {}
        }
    }

    pub fn get_last_input(&self) -> InputCode {
        unsafe{ mem::transmute(self.last_input) }
    }

    pub fn get_cursor_pos(&self) -> Vec2<f32> {
        self.cursor_pos
    }

    pub fn get_absolute_cursor_pos(&self) -> Vec2<f32> {
        self.absolute_cursor_pos
    }

    pub fn add_event(&mut self, name: &'static str, key: InputCode) {
        if !self.events.contains_key(name) {
            self.events.insert(name, InputEvent::new(key));
        }
    }

    pub fn remove_event(&mut self, name: &'static str) {
        if self.events.contains_key(name) {
            self.events.remove(name);
        }
    }

    pub fn clear(&mut self) {
        self.events.clear();
    }

    pub fn get_event(&mut self, name: &'static str) -> &mut InputEvent {
        return self.events.get_mut(name).unwrap()
    }

    pub fn is_touch(&self) -> bool {
        self.touch
    }

    pub fn is_touched(&self, position: &Position) -> bool {
        false
    }
}
