use image::GenericImageView;
use anyhow::*;

use wgpu::{
    Device,
    Queue,
    Extent3d,
    TextureDescriptor,
    TextureDimension,
    TextureUsages,
    TextureView,
    TextureViewDescriptor,
    AddressMode,
    FilterMode,
    Origin3d,
    TextureAspect,
    TextureFormat,
    Sampler,
    SamplerDescriptor,
    ImageDataLayout,
    ImageCopyTexture,
    BindGroupLayout,
    BindGroup,
};


pub struct Texture {
    bind_group: BindGroup,
    texture: wgpu::Texture,
    view: TextureView,
    sampler: Sampler,
}

impl Texture {
    pub fn new(device: &Device, queue: &Queue, layout: &BindGroupLayout, bytes: &[u8], label: Option<&str>) -> Result<Self> {
        let img = image::load_from_memory(bytes).unwrap();

        let rgba = img.as_rgba8().unwrap();
        let dimensions = img.dimensions();

        let size = Extent3d {
            width: dimensions.0,
            height: dimensions.1,
            depth_or_array_layers: 1,
        };
        let texture = device.create_texture(
            &TextureDescriptor {
                label,
                size,
                mip_level_count: 1,
                sample_count: 1,
                dimension: TextureDimension::D2,
                format: TextureFormat::Rgba8UnormSrgb,
                usage: TextureUsages::TEXTURE_BINDING | TextureUsages::COPY_DST,
            }
        );

        queue.write_texture(
            ImageCopyTexture {
                aspect: TextureAspect::All,
                texture: &texture,
                mip_level: 0,
                origin: Origin3d::ZERO,
            },
            rgba,
            ImageDataLayout {
                offset: 0,
                bytes_per_row: std::num::NonZeroU32::new(4 * dimensions.0),
                rows_per_image: std::num::NonZeroU32::new(dimensions.1),
            },
            size,
        );

        let view = texture.create_view(&TextureViewDescriptor::default());
        let sampler = device.create_sampler(
            &SamplerDescriptor {
                address_mode_u: AddressMode::ClampToEdge,
                address_mode_v: AddressMode::ClampToEdge,
                address_mode_w: AddressMode::ClampToEdge,
                mag_filter: FilterMode::Nearest,
                min_filter: FilterMode::Nearest,
                mipmap_filter: FilterMode::Nearest,
                ..Default::default()
            }
        );

        let bind_group = device.create_bind_group(
            &wgpu::BindGroupDescriptor {
                layout: &layout,
                entries: &[
                    wgpu::BindGroupEntry {
                        binding: 0,
                        resource: wgpu::BindingResource::TextureView(&view),
                    },
                    wgpu::BindGroupEntry {
                        binding: 1,
                        resource: wgpu::BindingResource::Sampler(&sampler),
                    }
                ],
                label: Some("texture_bind_group"),
            }
        );
        
        
        Ok(Self { bind_group, texture, view, sampler })
    }

    pub fn get_bind_group(&self) -> &BindGroup {
        &self.bind_group
    }

}
