use std::ops::Range;

use wgpu::{
    RenderPass,
    IndexFormat::Uint16,
    Buffer,
};

use crate::{
    Model,
    Texture,
    Resources,
    RenderData
};

static mut INDEX_CACHE: u32 = 0;

pub trait Renderer<'a> {
    fn render_instanced(&mut self, instances: Range<u32>);
    fn bind_model(&mut self, model: &'a Model);
    fn bind_texture(&mut self, texture: &'a Texture);
    fn bind_position(&mut self, position: &'a Buffer);
    fn bind_camera_buffer(&mut self, camera: &'a RenderData);
    fn render_texture(&mut self, resources: &'a Resources, model: &'a Model, texture: &'a Texture);
}

impl <'a, 'b> Renderer<'b> for RenderPass<'a>
where 'b: 'a {

    fn render_instanced(&mut self, instances: Range<u32>) {
        unsafe {
            self.draw_indexed(0..INDEX_CACHE, 0, instances)
        }
    }

    fn bind_model(&mut self, model: &'a Model) {
        self.set_vertex_buffer(0, model.get_vertex_buffer().slice(..));
        self.set_index_buffer(model.get_index_buffer().slice(..), Uint16);
        unsafe {
            INDEX_CACHE = model.get_amount_of_indices();
        }
    }

    fn bind_texture(&mut self, texture: &'a Texture) {
        self.set_bind_group(0, texture.get_bind_group(), &[]);
    }

    fn bind_camera_buffer(&mut self, render_data: &'a RenderData) {
        self.set_bind_group(1, render_data.get_camera_bind_group(), &[])
    }

    fn bind_position(&mut self, position: &'a Buffer) {
        self.set_vertex_buffer(1, position.slice(..));
    }

    fn render_texture(&mut self, resources: &'a Resources, model: &'a Model, texture: &'a Texture) {
        self.set_pipeline(resources.get_pipeline("texture_instance"));
        self.bind_model(model);
        self.bind_texture(texture);
    }
}


    // pub fn render_color(&self, model: &Model, color: &Color) {
    //     model.bind();
    //     self.color_program.bind();
    //     self.color_program.set_uniform_mat4f("u_MVP\0", self.vp.as_ptr());
    //     self.color_program.set_color("u_Color\0", color);
    //     unsafe {
    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }

        
    // pub fn render_texture(&self, model: &Model, texture: &Texture) {
    //     model.bind();
    //     texture.bind();
    //     self.texture_program.bind();
    //     self.texture_program.set_uniform_1i("u_Texture\0", 0);
    //     self.texture_program.set_uniform_mat4f("u_MVP\0", &self.vp[0]);
    //     unsafe {
    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }


    // pub fn render_colored_texture(&self, model: &Model, texture: &Texture, color: &Color) {
    //     model.bind();
    //     texture.bind();
    //     self.colored_texture_program.bind();
    //     self.colored_texture_program.set_uniform_1i("u_Texture\0", 0);
    //     self.colored_texture_program.set_uniform_mat4f("u_MVP\0", &self.vp[0]);
    //     self.colored_texture_program.set_color("u_Color\0", color);
    //     unsafe {
    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }

    // pub fn render_rainbow(&self, model: &Model, total_time: f32) {
    //     model.bind();
    //     self.rainbow_program.bind();
    //     self.rainbow_program.set_uniform_mat4f("u_MVP\0", &self.vp[0]);
    //     self.rainbow_program.set_uniform_1f("u_TotalTime\0", total_time);
    //     unsafe {
    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }

    // pub fn render_croppped_texture(&self, model: &Model, tex: &Texture, top_left: &Vec2f, top_right: &Vec2f) {
    //     model.bind();
    //     tex.bind();
    //     self.crop_program.bind();
    //     self.crop_program.set_uniform_mat4f("u_MVP\0", &self.vp[0]);
    //     self.crop_program.set_uniform_4f("u_NewCoords\0", top_left, top_right);
    //     unsafe {pub mod pipeline_data;

    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }


    // pub fn render_circle_color(&self, model: &Model, color: &Color) {
    //     model.bind();
    //     self.circle_color_program.bind();
    //     self.circle_color_program.set_uniform_mat4f("u_MVP\0", &self.vp[0]);
    //     self.circle_color_program.set_color("u_Color\0", color);
    //     unsafe {
    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }

    // pub fn render_circle_texture(&self, model: &Model, texture: &Texture) {
    //     model.bind();
    //     texture.bind();
    //     self.circle_texture_program.bind();
    //     self.circle_texture_program.set_uniform_1i("u_Texture\0", 0);
    //     self.circle_texture_program.set_uniform_mat4f("u_MVP\0", &self.vp[0]);
    //     unsafe {
    //         gl::DrawElements(gl::TRIANGLES, model.get_amount_of_indices() as i32, gl::UNSIGNED_INT, std::ptr::null());
    //     }
    // }
