use wgpu::{
    Device,
    Queue,
    Surface,
    SurfaceConfiguration,
    Buffer,
    PresentMode,
    util::DeviceExt,
    BindGroup
};

use crate::{
    Matrix4x4,
    Resources
};

pub struct RenderData {
    device: Device,
    queue: Queue,
    surface: Surface,
    config: SurfaceConfiguration,
    camera_buffer: Buffer,
    camera_bind_group: BindGroup,
}

impl RenderData {
    pub fn new(resources: &Resources, device: Device, queue: Queue, surface: Surface, config: SurfaceConfiguration) -> Self {
        let camera_buffer = device.create_buffer_init(
            &wgpu::util::BufferInitDescriptor {
                label: Some("Position Buffer"),
                contents: bytemuck::cast_slice(Matrix4x4::empty().as_ref()),
                usage: wgpu::BufferUsages::UNIFORM | wgpu::BufferUsages::COPY_DST
            }
        );

        let camera_bind_group = device.create_bind_group(&wgpu::BindGroupDescriptor {
            layout: &resources.get_bind_group_layout("matrix"),
            entries: &[
                wgpu::BindGroupEntry {
                    binding: 0,
                    resource: camera_buffer.as_entire_binding(),
                }
            ],
            label: Some("camera_bind_group"),
        });
        Self {
            queue,
            surface,
            config,
            device,
            camera_buffer,
            camera_bind_group
        }
    }

    pub fn update_camera_buffer(&self, data: &[f32; 16]) {
        self.queue.write_buffer(&self.camera_buffer, 0, bytemuck::cast_slice(data));
    }

    pub fn get_camera_bind_group(&self) -> &BindGroup {
        &self.camera_bind_group
    }
    
    pub fn get_device(&self) -> &Device {
        &self.device
    }

    pub fn get_queue(&self) -> &Queue {
        &self.queue
    }

    pub fn get_config(&self) -> &SurfaceConfiguration {
        &self.config
    }

    pub fn get_config_mut(&mut self) -> &mut SurfaceConfiguration {
        &mut self.config
    }
    
    pub fn get_surface(&self) -> &Surface {
        &self.surface
    }

    pub fn set_vsync(&mut self, value: bool) {
        if value {
            self.config.present_mode = PresentMode::Fifo;
        } else {
            self.config.present_mode = PresentMode::Mailbox;
        }
    }
}
