use wgpu::{
    Buffer,
    Device,
    util::DeviceExt,
};

use crate::{
    Vertex,
    Vec2,
    RenderData
};

const DEFAULT_INDICES: [u16; 6] = [
    0, 1, 2, 2, 3, 0
];


pub struct Model {
    amount_of_vertices:     u32,
    amount_of_indices:      u32,
    pub vertex_buffer:      Buffer,
    index_buffer:           Buffer
}

impl Model {

    pub fn new(render_data: &RenderData, vertices: &[Vertex], indices: &[u16]) -> Self {
        Model {
            amount_of_vertices: vertices.len() as u32,
            amount_of_indices: indices.len() as u32,
            vertex_buffer: Self::new_vertexbuffer(render_data.get_device(), vertices),
            index_buffer: Self::new_indexbuffer(render_data.get_device(), indices),
        }
    }

    pub fn new_box(render_data: &RenderData, dim: Vec2<f32>) -> Self {
        let vertices = Self::create_vertices(dim);
        Model {
            amount_of_vertices: vertices.len() as u32,
            amount_of_indices: DEFAULT_INDICES.len() as u32,
            vertex_buffer: Self::new_vertexbuffer(render_data.get_device(), &vertices),
            index_buffer: Self::new_indexbuffer(render_data.get_device(), &DEFAULT_INDICES),
        }
    }

    pub fn get_vertex_buffer(&self) -> &Buffer {
        &self.vertex_buffer
    }

    pub fn get_index_buffer(&self) -> &Buffer {
        &self.index_buffer
    }

    pub fn get_amount_of_indices(&self) -> u32 {
        self.amount_of_indices
    }

    pub fn get_amount_of_vertices(&self) -> u32 {
        self.amount_of_vertices
    }

    fn new_indexbuffer(device: &Device, indices: &[u16]) -> Buffer {
        device.create_buffer_init(
            &wgpu::util::BufferInitDescriptor {
                label: Some("Index Buffer"),
                contents: bytemuck::cast_slice(indices),
                usage: wgpu::BufferUsages::INDEX,
            }
        )
    }
    
    fn new_vertexbuffer(device: &Device, vertices: &[Vertex]) -> Buffer {
        device.create_buffer_init(
            &wgpu::util::BufferInitDescriptor {
                label: Some("Vertex Buffer"),
                contents: bytemuck::cast_slice(vertices),
                usage: wgpu::BufferUsages::VERTEX,
            }
        )
    }

    fn create_vertices(dim: Vec2<f32>) -> [Vertex; 4] {
        [
            Vertex::new(-dim.x/2.0, -dim.y/2.0, 0.0, 1.0),
            Vertex::new(dim.x/2.0, -dim.y/2.0, 1.0, 1.0),
            Vertex::new(dim.x/2.0, dim.y/2.0, 1.0, 0.0),
            Vertex::new(-dim.x/2.0, dim.y/2.0, 0.0, 0.0),
        ]
    }
}
