use std::collections::HashMap;

use wgpu::{
    Device,
    BindGroupLayout,
    RenderPipeline,
    SurfaceConfiguration,
    ShaderModuleDescriptor,
    include_wgsl,
};

use crate::{
    Model,
    Vertex,
    Texture,
    RenderData,
};

pub struct Resources{
    models: HashMap<&'static str, Model>,
    textures: HashMap<&'static str, Texture>,
    pipe_lines: HashMap<&'static str, RenderPipeline>,
    bind_group_layouts: HashMap<&'static str, BindGroupLayout>,
}

impl <'a>Resources {

    pub fn new() -> Self {
        Self {
            models: HashMap::new(),
            textures: HashMap::new(),
            pipe_lines: HashMap::new(),
            bind_group_layouts: HashMap::new()
        }
    }


    // Models

    pub fn get_model(&self, name: &'static str) -> &Model {
        match self.models.get(name) {
            Some(x) => &x,
            None => panic!("The model {} doesn't exist", name)
        }
    }

    pub fn add_model(&mut self, name: &'static str, model: Model) {
        if !self.models.contains_key(name) {
            self.models.insert(name, model);
        }
    }

    pub fn destroy_model(&mut self, name: &'static str) {
        if self.models.contains_key(name) {
            self.models.remove(name);
        }
    }

    pub fn clear_models(&mut self) {
        self.models.clear();
    }

    
    // Textures

    pub fn get_texture(&self, name: &'static str) -> &Texture {
        match self.textures.get(name) {
            Some(x) => &x,
            None => panic!("The texture {} doesn't exist", name)
        }
    }

    pub fn add_texture(&mut self, render_data: &RenderData, name: &'static str, bytes: &[u8]) {
        if !self.textures.contains_key(name) {
            self.textures.insert(name, Texture::new(render_data.get_device(), render_data.get_queue(), self.get_bind_group_layout("texture"), bytes, Some(name)).unwrap());
        }
    }

    pub fn destroy_texture(&mut self, name: &'static str) {
        if self.textures.contains_key(name) {
            self.textures.remove(name);
        }
    }

    pub fn clear_textures(&mut self) {
        self.textures.clear();
    }


    // Pipelines

    pub fn get_pipeline(&self, name: &'static str) -> &RenderPipeline {
        match self.pipe_lines.get(name) {
            Some(x) => &x,
            None => panic!("The pipe line {} doesn't exist", name)
        }
    }

    pub fn add_pipeline(&mut self, name: &'static str, pipeline: RenderPipeline) {
        if !self.pipe_lines.contains_key(name) {
            self.pipe_lines.insert(name, pipeline);
        }
    }

    pub fn destroy_pipeline(&mut self, name: &'static str) {
        if self.pipe_lines.contains_key(name) {
            self.pipe_lines.remove(name);
        }
    }


    // Bindgroups

    pub fn get_bind_group_layout(&self, name: &'static str) -> &BindGroupLayout {
        match self.bind_group_layouts.get(name) {
            Some(x) => &x,
            None => panic!("The bind group layout {} doesn't exist", name)
        }
    }

    pub fn add_bind_group_layout(&mut self, name: &'static str, bind_group_layout: BindGroupLayout) {
        if !self.bind_group_layouts.contains_key(name) {
            self.bind_group_layouts.insert(name, bind_group_layout);
        }
    }

    pub fn destroy_bind_group_layout(&mut self, name: &'static str) {
        if self.bind_group_layouts.contains_key(name) {
            self.bind_group_layouts.remove(name);
        }
    }



    pub fn create_default(&mut self, device: &Device, config: &SurfaceConfiguration) {

        self.add_bind_group_layout("texture", device.create_bind_group_layout(
            &wgpu::BindGroupLayoutDescriptor {
                entries: &[
                    wgpu::BindGroupLayoutEntry {
                        binding: 0,
                        visibility: wgpu::ShaderStages::FRAGMENT,
                        ty: wgpu::BindingType::Texture {
                            multisampled: false,
                            view_dimension: wgpu::TextureViewDimension::D2,
                            sample_type: wgpu::TextureSampleType::Float { filterable: true },
                        },
                        count: None,
                    },
                    wgpu::BindGroupLayoutEntry {
                        binding: 1,
                        visibility: wgpu::ShaderStages::FRAGMENT,
                        ty: wgpu::BindingType::Sampler(
                            wgpu::SamplerBindingType::Filtering,
                        ),
                        count: None,
                    },
                ],
                label: Some("texture_bind_group_layout"),
            }
        ));

        self.add_bind_group_layout("matrix", device.create_bind_group_layout(
            &wgpu::BindGroupLayoutDescriptor {
                entries: &[
                    wgpu::BindGroupLayoutEntry {
                        binding: 0,
                        visibility: wgpu::ShaderStages::VERTEX,
                        ty: wgpu::BindingType::Buffer {
                            ty: wgpu::BufferBindingType::Uniform,
                            has_dynamic_offset: false,
                            min_binding_size: None,
                        },
                        count: None,
                    }
                ],
                label: Some("matrix_bind_group_layout"),
            }
        ));
        
        self.add_pipeline("texture_instance", Self::new_pipeline(
            "Texture Instance",
            device,
            config,
            &[
                self.get_bind_group_layout("texture"),
                self.get_bind_group_layout("matrix"),
            ], 
            &include_wgsl!("../../res/shader/instance/texture.wgsl"),
        ));

        self.add_pipeline("texture_instance", Self::new_pipeline(
            "Texture Instance",
            device,
            config,
            &[
                self.get_bind_group_layout("texture"),
                self.get_bind_group_layout("matrix"),
            ], 
            &include_wgsl!("../../res/shader/instance/texture.wgsl"),
        ));
    }

    pub fn new_pipeline(name: &str, device: &Device, config: &SurfaceConfiguration, bind_group_layouts: &[&BindGroupLayout], 
                        shader_descriptor: &ShaderModuleDescriptor) -> RenderPipeline {

        let shader = device.create_shader_module(shader_descriptor);
    
        let render_pipeline_layout = device.create_pipeline_layout(&wgpu::PipelineLayoutDescriptor {
            label: Some(format!("{} Pipeline Layout", name).as_str()),
            bind_group_layouts: bind_group_layouts,
            push_constant_ranges: &[],
        });
    
    
        return device.create_render_pipeline(&wgpu::RenderPipelineDescriptor {
            label: Some(format!("{} Pipeline", name).as_str()),
            layout: Some(&render_pipeline_layout),
            vertex: wgpu::VertexState {
                module: &shader,
                entry_point: "vs_main",
                buffers: &[
                    Vertex::desc(),
                    Vertex::instance_desc()
                ],
            },
            fragment: Some(wgpu::FragmentState {
                module: &shader,
                entry_point: "fs_main",
                targets: &[wgpu::ColorTargetState {
                    format: config.format,
                    blend: Some(wgpu::BlendState::ALPHA_BLENDING),
                    write_mask: wgpu::ColorWrites::ALL,
                }],
            }),
            primitive: wgpu::PrimitiveState {
                topology: wgpu::PrimitiveTopology::TriangleList,
                strip_index_format: None,
                front_face: wgpu::FrontFace::Ccw,
                cull_mode: Some(wgpu::Face::Back),
                polygon_mode: wgpu::PolygonMode::Fill,
                unclipped_depth: false,
                conservative: false,
            },
            depth_stencil: None,
            multisample: wgpu::MultisampleState {
                count: 1,
                mask: !0,
                alpha_to_coverage_enabled: false,
            },
            multiview: None,
        });
    }
}
