use crate::{
    Matrix4x4,
    Vec2
};

pub struct Position {
    pub pos: Vec2<f32>,
    pub scale: Vec2<f32>,
    pub size: Vec2<f32>,
    pub rot: f32,
    rect: (Vec2<f32>, Vec2<f32>), // top left, bottom right
    matrix: Matrix4x4,
    moved: bool,
}


impl Position {

    pub fn new(pos: Vec2<f32>, size: Vec2<f32>, scale: Vec2<f32>, rot: f32) -> Self {
        let mut buf = Self {
            pos,
            scale,
            size,
            rot,
            rect: (Vec2::new(0.0, 0.0), Vec2::new(0.0, 0.0)),
            matrix: Matrix4x4::empty(),
            moved: false,
        };
        buf.build();
        return buf;
    }

    pub fn check_offscreen(&self, fov: Vec2<f32>) -> bool {
        self.rect.0.x > fov.x ||
        self.rect.1.x < -fov.x ||
        self.rect.0.y < -fov.y ||
        self.rect.1.y > fov.y
    }

    pub fn get_matrix(&self) -> Matrix4x4 {
        self.matrix
    }


    pub fn set(&mut self, new_pos: Vec2<f32>, new_scale: Vec2<f32>, new_rot: f32) {
        self.pos = new_pos;
        self.scale = new_scale;
        self.rot = new_rot;
        self.build();
    }

    pub fn begin(&mut self) {
        self.moved = false;
    }

    pub fn build(&mut self) {
        self.moved = true;
        self.rect = (
            Vec2::new(self.pos.x - self.size.x * self.scale.x, self.pos.y + self.size.y * self.scale.y),
            Vec2::new(self.pos.x + self.size.x * self.scale.x, self.pos.y - self.size.y * self.scale.y),
        );
        self.matrix = Matrix4x4::from_translate(self.pos);
        if self.rot != 0.0 {
            self.matrix.rotate(self.rot);
        }
        if self.scale != Vec2::new(1.0, 1.0) {
            self.matrix.scale(self.scale);
        }
    }

    pub fn has_moved(&self) -> bool {
        self.moved
    }

    pub fn after_move(&mut self) {
        self.moved = false;
    }
}
