use rand::Rng;
use enschin::{
    Enschin, 
    EnschinContext, 
    Vec2,
    SceneController, 
    SceneContext,
    Component,
    Position,
    Update,
    RenderType,
    Model,
    Render,
    Renderer,
    RenderPass,
    InputCode
};

const GRAVITY: f32 = -2.5;

#[cfg_attr(target_os = "android", ndk_glue::main(backtrace = "on"))]
pub fn main() {
    Enschin::init("Bun", GameScene{total_bunnies: 2});
}

pub struct GameScene {
    total_bunnies: u32,
}

impl SceneController for GameScene {
    fn setup(&mut self, scene: &mut SceneContext, enschin: &mut EnschinContext) {
        let model = Model::new_box(enschin.render_data, Vec2::new(0.39, 0.555));
        enschin.resources.add_model("hel", model);
        enschin.resources.add_texture(enschin.render_data, "tree", include_bytes!("./img/wabbit.png"));

        for _ in 0..self.total_bunnies {
            let bunny = Bunny::new(scene, enschin);
            scene.add_component(bunny);
        }

        enschin.clear_color.b = 1.0;

        scene.input.add_event("spawn_many", InputCode::MouseLeft);
        scene.input.add_event("spawn_one", InputCode::MouseRight);

        scene.camera.set_fov(7.0);
    }
    

    fn dispose(&mut self, _scene: &mut SceneContext, enschin: &mut EnschinContext) {
        enschin.resources.destroy_model("hel");
        enschin.resources.destroy_texture("tree");
    }

    fn update(&mut self, scene: &mut SceneContext, enschin: &mut EnschinContext) {
        if scene.input.get_event("spawn_many").held  || scene.input.is_touch() {
            for _ in 0..500 {
                let bunny = Bunny::new(scene, enschin);
                scene.add_component(bunny);
                self.total_bunnies+=1;
            }
            println!("There are now {} bunnies!", self.total_bunnies);
        }
        if scene.input.get_event("spawn_one").held {
            let bunny = Bunny::new(scene, enschin);
            scene.add_component(bunny);
            self.total_bunnies+=1;
            println!("There are now {} bunnies!", self.total_bunnies);
        }
    }

    fn name(&self) -> &'static str {
        "Oof"
    }
}

#[derive(Component)]
#[impls(Update, Render)]
pub struct Bunny {
    position: Position,
    speed: Vec2<f32>,
}

impl Bunny {
    pub fn new(scene: &mut SceneContext, _enschin: &mut EnschinContext) -> Self {
        let speed = Vec2::new(
            rand::thread_rng().gen_range(0.0..5.0) - 2.5,
            rand::thread_rng().gen_range(0.0..15.0) - 7.5
        );
        Bunny {
            position: Position::new(scene.input.get_cursor_pos(), Vec2::new(0.39, 0.555), Vec2::new(1.0, 1.0), 0.0),
            speed: speed,
        }
    }
}

impl Render for Bunny {
    fn render_type(&self) -> RenderType {
        RenderType::Multi
    }
    fn always_render(&self) -> bool {
        true
    }
    fn get_pos(&self) -> Option<&Position> {
        Some(&self.position)
    }
    fn render_multi<'a>(&'a self, enschin: &'a EnschinContext, render_pass: &mut RenderPass<'a>) {
        render_pass.render_texture(
            enschin.resources, 
            enschin.resources.get_model("hel"), 
            enschin.resources.get_texture("tree")
        );
    }
}

impl Update for Bunny {
    fn update(&mut self, scene: &mut SceneContext, enschin: &mut EnschinContext) {
        self.position.pos += self.speed * enschin.delta_time;
        self.speed.y += GRAVITY * enschin.delta_time;
        let fov = scene.camera.get_fov();

        if self.position.pos.x >= fov.x {
            self.speed.x *= -1.0;
            self.position.pos.x = fov.x;
        } else if self.position.pos.x <= -fov.x {
            self.speed.x *= -1.0;
            self.position.pos.x = -fov.x;
        }
    
        if self.position.pos.y < -fov.y {
            self.speed.y *= -0.85;
            self.position.pos.y = -fov.y;
            if rand::thread_rng().gen_range(0.0..1.0) > 0.5{
                self.speed.y -= rand::thread_rng().gen_range(0.0..15.0);
            }
        } else if self.position.pos.y > fov.y {
            self.speed.y = 0.0;
            self.position.pos.y = fov.y;
        }
        self.position.build();
    }
}
