extern crate gl;
extern crate glfw;
use glfw::{Context};

use crate::math::vec2i::Vec2i;
use crate::graphics::color::Color;

#[derive(Eq, PartialEq)]
pub enum RenderCallback {
    Close,
    Nothing,
    Resize
}

pub struct Window {
    size:           Vec2i,
    fullscreen:     bool,
    vsync:          bool,
    antialiasing:   bool,
    should_close:   bool,
    glfw:           glfw::Glfw,
    glfw_window:    glfw::Window,
}

impl Window {
    pub fn new(title: &str, size: &Vec2i) -> Self {

        let glfw = glfw::init(glfw::FAIL_ON_ERRORS).unwrap();
        let mut window = Window {
            size:           *size,
            fullscreen:     false,
            vsync:          true,
            antialiasing:   true,
            should_close:   false,
            glfw:           glfw,
            glfw_window:    glfw.create_window(size.x as u32, size.y as u32, title, glfw::WindowMode::Windowed).expect("Failed to create GLFW window.").0
        };

        glfw::WindowHint::ContextVersionMajor(3);
        glfw::WindowHint::ContextVersionMinor(3);
        glfw::WindowHint::OpenGlForwardCompat(true);
        glfw::WindowHint::AutoIconify(false);
        glfw::WindowHint::Samples(4);

        window.glfw_window.make_current();
        window.glfw_window.set_key_polling(true);
    
        gl::load_with(
            |s| window.glfw.get_proc_address_raw(s)
        );

        window.on_size_change(size);
        window.set_vsync(true);
        window.set_antialiasing(true);

        return window;
    }

    pub fn update(&mut self) -> RenderCallback {
        self.glfw.poll_events();
        let (w, h) = self.glfw_window.get_size();
        if self.glfw_window.should_close() {
            return RenderCallback::Close;
        } else if (Vec2i{x: w, y: h}) != self.size {
            return RenderCallback::Resize;
        } else {
            return RenderCallback::Nothing;
        }
    }

    pub fn swap_buffers(&mut self) {
        unsafe {
            gl::Clear(gl::COLOR_BUFFER_BIT);
        }
        self.glfw_window.swap_buffers();
    }

    fn on_size_change(&mut self, new_size: &Vec2i) {
        self.size = *new_size;
        unsafe {
            gl::Viewport(0, 0, new_size.x, new_size.y);
        }
    }

    pub fn get_size(&self) -> Vec2i{
        self.size
    }

    pub fn set_size(&mut self, new_size: &Vec2i) {
        self.size = *new_size;
        self.glfw_window.set_size(new_size.x as i32, new_size.y as i32);
    }

    pub fn set_title(&mut self, new_title: &str) {
        self.glfw_window.set_title(new_title);
    }

    pub fn is_vsync(&self) -> bool {
        self.vsync
    }

    pub fn set_vsync(&mut self, state: bool) {
        self.vsync = state;
        if state {
            self.glfw.set_swap_interval(1);
        } else {
            self.glfw.set_swap_interval(0);
        }

    }

    pub fn is_antialiasing(&self) -> bool {
        self.antialiasing
    }

    pub fn set_antialiasing(&mut self, state: bool) {
        self.antialiasing = state;
        unsafe {
            if state {
                gl::Enable(gl::MULTISAMPLE);
            } else {
                gl::Enable(gl::MULTISAMPLE);
            }
        }
    }

    pub fn set_icon(&mut self, icon_path: &String) {
        // TODO
    }

    pub fn set_background_color(&self, color: &Color) {
        unsafe {
            gl::ClearColor(color.r, color.g, color.b, color.a);
        }
    }

    pub fn is_fullscreen(&self) -> bool {
        self.fullscreen
    }

    pub fn set_fullscreen(&mut self, fullscreen: bool) {
        self.fullscreen = fullscreen;
        // TODO
    }
}