use crate::math::vec3f::Vec3f;

fn length(v: &Vec3f) -> f32 {
    let l: f32 = v.x * v.x + v.y * v.y + v.z * v.z;
    return l.sqrt();
}


fn multiply(result: &mut[f32; 16], lhs: [f32; 16], rhs: [f32; 16]) {
    for i in (0..16).step_by(4) {
        for j in (0..16).step_by(4) {
            result[i+j] = lhs[i] * rhs[j] +
            lhs[i+1] * rhs[4+j] +
            lhs[i+2] * rhs[8+j] +
            lhs[i+3] * rhs[12+j];
        }
    }
}


fn translate(m: &mut[f32; 16], pos: &Vec3f) {
    for i in (0..16).step_by(4) {
        m[12 + i] += m[i] * pos.x + m[4 + i] * pos.y + m[8 + i] * pos.z;
    }
}


fn rotate(m: &mut[f32; 16], a: f32, axis: &Vec3f) {
    let mut s_temp: [f32; 16] = Default::default();
    set_rotate(&mut s_temp, a, axis);
    let copy = s_temp;
    multiply(&mut s_temp, copy, *m);
    *m = s_temp;
}

fn set_rotate(rm: &mut[f32; 16], a: f32, axis: &Vec3f) {
    rm[3] = 0.0;
    rm[7] = 0.0;
    rm[11]= 0.0;
    rm[12]= 0.0;
    rm[13]= 0.0;
    rm[14]= 0.0;
    rm[15]= 1.0;
    let s = -a.sin();
    let c = -a.cos();
    if (Vec3f{x: 1.0, y: 0.0, z: 0.0}) == *axis {
        rm[5] = c;     rm[10]= c;
        rm[6] = s;     rm[9] = -s;
        rm[1] = 0.0;   rm[2] = 0.0;
        rm[4] = 0.0;   rm[8] = 0.0;
        rm[0] = 1.0;
    } else if (Vec3f{x: 0.0, y: 1.0, z: 0.0}) == *axis {
        rm[0] = c;     rm[10]= c;
        rm[8] = s;     rm[2] = -s;
        rm[1] = 0.0;   rm[4] = 0.0;
        rm[6] = 0.0;   rm[9] = 0.0;
        rm[5] = 1.0;
    } else if (Vec3f{x: 0.0, y: 0.0, z: 1.0}) == *axis {
        rm[0] = c;     rm[5] = c;
        rm[1] = s;     rm[4] = -s;
        rm[2] = 0.0;   rm[6] = 0.0;
        rm[8] = 0.0;   rm[9] = 0.0;
        rm[10]= 1.0;
    } else {
        let len = length(axis);
        let mut ax = *axis;
        if 1.0 != len {
            let recip_len: f32 = 1.0 / len;
            ax *= recip_len;
        }
        let nc = 1.0 - c;
        let xy = ax.x * ax.y;
        let yz = ax.y * ax.z;
        let zx = ax.z * ax.x;
        let xs = ax.x * s;
        let ys = ax.y * s;
        let zs = ax.z * s;
        rm[0] = ax.x*ax.x*nc +  c;
        rm[4] =  xy*nc - zs;
        rm[8] =  zx*nc + ys;
        rm[1] =  xy*nc + zs;
        rm[5] = ax.y*ax.y*nc +  c;
        rm[9] =  yz*nc - xs;
        rm[2] =  zx*nc - ys;
        rm[6] =  yz*nc + xs;
        rm[10] = ax.z*ax.z*nc +  c;
    }
}

fn scale(m: &mut[f32; 16], scaling: &Vec3f) {
    for i in (0..16).step_by(4) {
        m[     i] *= scaling.x;
        m[ 4 + i] *= scaling.y;
        m[ 8 + i] *= scaling.z;
    }
}

fn frustum(m: &mut[f32; 16], left: f32, right: f32, bottom: f32, top: f32, near: f32, far: f32) {
    let r_width  = 1.0 / (right - left);
    let r_height = 1.0 / (top - bottom);
    let r_depth  = 1.0 / (near - far);
    let x = 2.0 * (near * r_width);
    let y = 2.0 * (near * r_height);
    let a = (right + left) * r_width;
    let b = (top + bottom) * r_height;
    let c = (far + near) * r_depth;
    let d = 2.0 * (far * near * r_depth);
    m[0] = x;
    m[5] = y;
    m[8] = a;
    m[ 9] = b;
    m[10] = c;
    m[14] = d;
    m[11] = -1.0;
    m[ 1] = 0.0;
    m[ 2] = 0.0;
    m[ 3] = 0.0;
    m[ 4] = 0.0;
    m[ 6] = 0.0;
    m[ 7] = 0.0;
    m[12] = 0.0;
    m[13] = 0.0;
    m[15] = 0.0;
}

fn ortho(m: &mut[f32; 16], left: f32, right: f32, bottom: f32, top: f32, near: f32, far: f32) {
    let r_width  = 1.0 / (right - left);
    let r_height = 1.0 / (top - bottom);
    let r_depth  = 1.0 / (far - near);
    let x =  2.0 * (r_width);
    let y =  2.0 * (r_height);
    let z = -2.0 * (r_depth);
    let tx = -(right + left) * r_width;
    let ty = -(top + bottom) * r_height;
    let tz = -(far + near) * r_depth;
    m[0] = x;
    m[5] = y;
    m[10] = z;
    m[12] = tx;
    m[13] = ty;
    m[14] = tz;
    m[15] = 1.0;
    m[1] = 0.0;
    m[2] = 0.0;
    m[3] = 0.0;
    m[4] = 0.0;
    m[6] = 0.0;
    m[7] = 0.0;
    m[8] = 0.0;
    m[9] = 0.0;
    m[11] = 0.0;
}

fn set_look_at(rm: &mut[f32; 16], eye: &Vec3f, center: &Vec3f, up: &Vec3f) {
    // See the OpenGL GLUT documentation for gluLookAt for a description
    // of the algorithm. We implement it in a straightforward way:
    let mut f = *center - *eye;
    // Normalize f
    let rlf: f32 = 1.0 / length(&f);
    f*=rlf;
    // compute s = f x up (x means "cross product")
    let mut s = Vec3f {
        x: f.x * up.z - f.z * up.y,
        y: f.y * up.x - f.x * up.z,
        z: f.z * up.y - f.y * up.x,
    };
    
    // and normalize s
    let rls: f32 = 1.0 / length(&s);
    s *= rls;
    // compute u = s x f
    let u = Vec3f {
        x: s.y * f.z - s.z * f.y,
        y: s.z * f.x - s.x * f.z,
        z: s.x * f.y - s.y * f.x,
    };

    rm[0] = s.x;
    rm[1] = u.x;
    rm[2] = -f.x;
    rm[3] = 0.0;
    rm[4] = s.y;
    rm[5] = u.y;
    rm[6] = -f.y;
    rm[7] = 0.0;
    rm[8] = s.z;
    rm[9] = u.z;
    rm[10] = -f.z;
    rm[11] = 0.0;
    rm[12] = 0.0;
    rm[13] = 0.0;
    rm[14] = 0.0;
    rm[15] = 1.0;
    translate(rm, &-*eye);
}

fn print_matrix(matrix: &mut[f32; 16]) {
    println!("Matrix: ");
    for i in (0..16).step_by(4) {
        println!("{}\t{}\t{}\t{}", matrix[i*4], matrix[i*4+1], matrix[i*4+2], matrix[i*4+3]);
    }
}

fn print_matrix_as_wurst(matrix: &mut[f32; 16]) {
    println!("Matrix as Wurst: ");
    for i in (0..16).step_by(4) {
        print!("{}", matrix[i]);
    }
    println!("");
}