use std::fs;

extern crate gl;

use crate::math::vec2f::Vec2f;
use crate::graphics::color::Color;

// TODO
// Shader struct with id and uniform location id vec as member

// pub fn create_program(fragment_path: &str, vertex_path: &str) -> u32 {
        
// }


pub enum ShaderType {
    VERTEX,
    FRAGMENT,
}

impl ShaderType {
    fn value(&self) -> u32 {
        match *self {
            ShaderType::VERTEX => gl::VERTEX_SHADER,
            ShaderType::FRAGMENT => gl::FRAGMENT_SHADER,
        }
    }
}

pub fn link_shader_program(vertex_shader: u32, fragment_shader: u32) -> u32 {
    // Create the shader program
    let program: u32;
    let program_linked: i32 = 0;

    unsafe {
        program = gl::CreateProgram();

        gl::AttachShader(program, vertex_shader);
        gl::AttachShader(program, fragment_shader);

        gl::LinkProgram(program);
        gl::GetProgramiv(program, gl::LINK_STATUS, program_linked as *mut gl::types::GLint);
    }

    if program_linked != gl::TRUE as i32 {
        let mut buf = Vec::with_capacity(program_linked as usize);
        let buf_ptr = buf.as_mut_ptr() as *mut gl::types::GLchar;
        unsafe {
            gl::GetProgramInfoLog(program, program_linked, std::ptr::null_mut(), buf_ptr);
            buf.set_len(program_linked as usize);
        };

        println!("Failed to link shader: ");
        println!("{}", match String::from_utf8(buf) {
            Ok(log) => log,
            Err(vec) => panic!("Could not convert link log from buffer: {}", vec)
        });
    }

    unsafe {
        gl::ValidateProgram(program);
    }

    return program;
}

pub fn compile_shader(shader_path: &str, shader_type: u32) -> u32 {
    // Read the shader code from files 
    let shader_code = fs::read_to_string(shader_path);
    let shader_code = match shader_code {
        Ok(string) => string,
        Err(error) => panic!("Could not open the shader file: {:?}", error)
    };

    // Create the shader
    let id: u32;
    unsafe {
        id = gl::CreateShader(shader_type);
    }

    // Compile the shader
    unsafe {
        gl::ShaderSource(id, 1, shader_code.as_ptr() as *const *const gl::types::GLchar, 0 as *const gl::types::GLint);
        gl::CompileShader(id);
    }

    // Error handling
    let mut result: gl::types::GLint = 0;
    unsafe {
        gl::GetShaderiv(id, gl::COMPILE_STATUS, &mut result);
    }

    println!("{} shader compile status: {}", if shader_type == gl::VERTEX_SHADER { "".to_string() } else { "fragment".to_string() }, result);
    if result == gl::FALSE as i32 {
        let length: i32 = 0;
        unsafe {
            gl::GetShaderiv(id, gl::INFO_LOG_LENGTH, length as *mut i32);
        }

        let mut buf = Vec::with_capacity(length as usize);
        let buf_ptr = buf.as_mut_ptr() as *mut gl::types::GLchar;
        unsafe {
            gl::GetProgramInfoLog(id, length, std::ptr::null_mut(), buf_ptr);
            buf.set_len(length as usize);
        };

        println!("Failed to compile {} shader: ", if shader_type == gl::VERTEX_SHADER { "".to_string() } else { "fragment".to_string() });
        println!("{}", match String::from_utf8(buf) {
            Ok(log) => log,
            Err(vec) => panic!("Could not convert compile log from buffer: {}", vec)
        });
        unsafe {
            gl::DeleteShader(id);
        }
        return 0;
    }
    return id;
}

pub fn bind(id: u32) {
    unsafe {
        gl::UseProgram(id);
    }
}

pub fn ubnind(id: u32) {
    unsafe {
        gl::UseProgram(id);
    }
}

pub fn delete_program(id: u32) {
    unsafe {
        gl::DeleteProgram(id);
    }
}

pub fn get_uniform_location(shader_id: u32, name: &String) -> i32 {
    unsafe {
        return gl::GetUniformLocation(shader_id, name.as_ptr() as *const gl::types::GLchar);
    }
}

pub fn set_uniform_1i(id: i32, value: i32) {
    unsafe {
        gl::Uniform1i(id, value);
    }
}

pub fn set_uniform_1f(id: i32, value: f32) {
    unsafe {
        gl::Uniform1f(id, value);
    }
}

pub fn set_uniform_2f(id: i32, v: &Vec2f) {
    unsafe {
        gl::Uniform2f(id, v.x, v.y);
    }
}

pub fn set_uniform_4f(id: i32, v1: &Vec2f, v2: &Vec2f) {
    unsafe {
        gl::Uniform4f(id, v1.x, v1.y, v2.x, v2.y);
    }
}

pub fn set_uniform_mat4f(id: i32, matrix: *const f32) {
    unsafe {
        gl::UniformMatrix4fv(id, 1, gl::FALSE, matrix);
    }
}

pub fn set_color(id: i32, color: &Color) { 
    unsafe {
        gl::Uniform4f(id, color.r, color.g, color.b, color.a);
    }
}

