use crate::math::vec2f::Vec2f;
use crate::graphics::buffers::vertex_array;
use crate::graphics::buffers::vertex_buffer;
use crate::graphics::buffers::vertex_buffer_layout;
use crate::graphics::buffers::index_buffer;


struct Model {
    amount_of_vertices:     u16,
    amount_of_indices:      u16,
    index_buffer_id:        u32,
    vertex_buffer_id:       u32,
    vertex_array_id:        u32
    // TODO box2dshape
}

impl Model {
    const DEFAULT_TEX_COORS: [f32; 8] = [
        0.0, 0.0,
        1.0, 0.0,
        1.0, 1.0,
        0.0, 1.0
    ];

    const DEFAULT_INDICES: [u16;6] = [
        0, 1, 2, 2, 3, 0
    ];


    pub fn new_box_model(dim: &Vec2f) -> Self {
        let vertices = Self::generate_vertecis_tex_coords(dim);

        let index_buffer_id = index_buffer::create_index_buffer(Self::DEFAULT_INDICES.as_ptr(), 4);
        let vertex_buffer_id = vertex_buffer::create_vertex_buffer(vertices.as_ptr(), 4*4*4);
        let vertex_array_id = vertex_array::create_vertex_array();

        let mut layout = vertex_buffer_layout::VertexBufferLayout::new();
        layout.add_float(2);
        layout.add_float(2);

        vertex_array::add_buffer_to_vertex_array(vertex_array_id, vertex_buffer_id, &layout);
        
        Self {
            amount_of_vertices:     4,
            amount_of_indices:      6,
            index_buffer_id:        index_buffer_id,
            vertex_buffer_id:       vertex_buffer_id,
            vertex_array_id:        vertex_array_id
        }
    }

    pub fn bind(&self) {
        index_buffer::delete_index_buffer(self.index_buffer_id);
        vertex_buffer::delete_vertex_buffer(self.vertex_buffer_id);
        vertex_array::delete_vertex_array(self.vertex_array_id);
    }


    fn generate_vertecis_tex_coords(dim: &Vec2f) -> [f32; 16] {
        [
            -dim.x/2.0,
            -dim.y/2.0,
            0.0,
            0.0,
            dim.x/2.0,
            -dim.y/2.0,
            1.0,
            0.0,
            dim.x/2.0,
            dim.y/2.0,
            1.0,
            1.0,
            -dim.x/2.0,
            dim.y/2.0,
            0.0,
            1.0
        ]
    }
}

impl Drop for Model {
    fn drop(&mut self) {
        index_buffer::delete_index_buffer(self.index_buffer_id);
        vertex_buffer::delete_vertex_buffer(self.vertex_buffer_id);
        vertex_array::delete_vertex_array(self.vertex_array_id);
    }
}


