use crate::hex::HexEncoder;
use crate::{encode_string_unchecked, Encoder, Error, StringEncoder};

/// Responsible for percent encoding data.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct PercentEncoder {
    hex_encoder: HexEncoder,
    no_encoding: AsciiSpecialSet,
}

impl PercentEncoder {
    //! Constructors

    /// Creates a new percent encoder. The `do_not_encode` characters should be unique US-ASCII special characters. All
    /// other characters in the slice will be ignored.
    pub fn new(do_not_encode: &[u8]) -> Self {
        Self {
            hex_encoder: HexEncoder::UPPERCASE_ENCODER,
            no_encoding: AsciiSpecialSet::new(do_not_encode),
        }
    }
}

impl PercentEncoder {
    //! Encoding Checks

    /// Checks if the character needs encoding.
    #[inline(always)]
    pub fn needs_encoding(&self, c: u8) -> bool {
        if c.is_ascii_alphanumeric() {
            false
        } else if !c.is_ascii() || c.is_ascii_control() {
            true
        } else {
            !self.no_encoding.contains(&c)
        }
    }
}

impl Encoder for PercentEncoder {
    fn encoded_len(&self, data: &[u8]) -> Result<usize, Error> {
        let encoded_count: usize = data.iter().filter(|c| self.needs_encoding(**c)).count();
        let extra_len: usize = encoded_count.checked_mul(2).ok_or(Error::IntegerOverflow)?;
        data.len()
            .checked_add(extra_len)
            .ok_or(Error::IntegerOverflow)
    }

    fn encode_slice(&self, data: &[u8], target: &mut [u8]) -> Result<usize, Error> {
        let encoded_len: usize = self.encoded_len(data)?;
        if target.len() < encoded_len {
            Err(Error::InsufficientTargetSpace)
        } else {
            let target: &mut [u8] = &mut target[..encoded_len];
            let mut t: usize = 0;
            for d in data {
                if self.needs_encoding(*d) {
                    target[t] = b'%';
                    t += 1;
                    let (a, b) = self.hex_encoder.encode(*d);
                    target[t] = a;
                    t += 1;
                    target[t] = b;
                    t += 1;
                } else {
                    target[t] = *d;
                    t += 1;
                }
            }
            Ok(encoded_len)
        }
    }
}

impl StringEncoder for PercentEncoder {
    fn encode_string(&self, data: &[u8], target: &mut String) -> Result<usize, Error> {
        unsafe { encode_string_unchecked(self, data, target) }
    }
}

/// Represents a set of special US-ASCII characters.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
struct AsciiSpecialSet {
    data: [u8; 34], // 33 special chars + len
}

impl AsciiSpecialSet {
    //! Constructors

    /// Creates a new set with the chars. The chars will be filtered and de-duplicated.
    pub fn new(chars: &[u8]) -> Self {
        let mut set: AsciiSpecialSet = AsciiSpecialSet { data: [0u8; 34] };
        for c in chars {
            if c.is_ascii() && !c.is_ascii_alphanumeric() && !c.is_ascii_control() {
                if !set.contains(c) {
                    set.add(c)
                }
            }
        }
        set
    }

    /// Adds the character. Does not validate the character.
    #[inline(always)]
    fn add(&mut self, c: &u8) {
        let last: usize = self.data.len() - 1;
        let len: usize = self.data[last] as usize;
        self.data[len] = *c;
        self.data[last] += 1;
    }
}

impl AsciiSpecialSet {
    //! Contains

    /// Checks if the character is in the set.
    #[inline(always)]
    pub fn contains(&self, c: &u8) -> bool {
        self.data[0..(self.data[self.data.len() - 1] as usize)].contains(c)
    }
}
