use std::marker::PhantomData;

use crate::hex::HexDecoder;
use crate::{Decoder, Error};

/// Responsible for decoding percent encoded data.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct PercentDecoder {
    _phantom: PhantomData<()>,
}

impl Default for PercentDecoder {
    fn default() -> Self {
        Self {
            _phantom: PhantomData::default(),
        }
    }
}

impl PercentDecoder {
    //! Decoding

    /// Counts the number of encoded bytes.
    pub fn encoded_count(&self, data: &[u8]) -> usize {
        match data.iter().position(|c| *c == b'%') {
            Some(p) => {
                if data.len() < 3 || p > data.len() - 3 {
                    0
                } else if !data[p + 1].is_ascii_hexdigit() || !data[p + 2].is_ascii_hexdigit() {
                    self.encoded_count(&data[p + 1..])
                } else {
                    1 + self.encoded_count(&data[p + 3..])
                }
            }
            None => 0,
        }
    }
}

impl Decoder for PercentDecoder {
    fn decoded_len(&self, data: &[u8]) -> Result<usize, Error> {
        Ok(data.len() - (2 * self.encoded_count(data)))
    }

    fn decode_slice(&self, data: &[u8], target: &mut [u8]) -> Result<usize, Error> {
        if data.len() == 0 {
            Ok(0)
        } else if data[0] == b'%' {
            if data.len() < 3 {
                if target.len() < data.len() {
                    Err(Error::InsufficientTargetSpace)
                } else {
                    (&mut target[..data.len()]).copy_from_slice(data);
                    Ok(data.len())
                }
            } else if target.len() < 1 {
                Err(Error::InsufficientTargetSpace)
            } else if !data[1].is_ascii_hexdigit() || !data[2].is_ascii_hexdigit() {
                target[0] = b'%';
                Ok(1 + self.decode_slice(&data[1..], &mut target[1..])?)
            } else {
                target[0] = HexDecoder::decode(data[1], data[2]);
                Ok(1 + self.decode_slice(&data[3..], &mut target[1..])?)
            }
        } else {
            match data.iter().position(|c| *c == b'%') {
                Some(p) => {
                    if target.len() < p {
                        Err(Error::InsufficientTargetSpace)
                    } else {
                        (&mut target[..p]).copy_from_slice(&data[..p]);
                        Ok(p + self.decode_slice(&data[p..], &mut target[p..])?)
                    }
                }
                None => {
                    if target.len() < data.len() {
                        Err(Error::InsufficientTargetSpace)
                    } else {
                        (&mut target[..data.len()]).copy_from_slice(&data);
                        Ok(data.len())
                    }
                }
            }
        }
    }
}
