use crate::{encode_string_unchecked, Encoder, Error, StringEncoder};

/// Responsible for encoding data as hex.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct HexEncoder {
    encoding_table: &'static [u8; 16],
}

impl HexEncoder {
    //! Special Encoders

    /// The lowercase encoder.
    pub const LOWERCASE_ENCODER: Self = Self {
        encoding_table: b"0123456789abcdef",
    };

    /// The uppercase encoder.
    pub const UPPERCASE_ENCODER: Self = Self {
        encoding_table: b"0123456789ABCDEF",
    };
}

impl HexEncoder {
    //! Encoding

    /// Encodes a single byte. Returns the (high, low) encoded bytes.
    #[inline(always)]
    pub fn encode(&self, b: u8) -> (u8, u8) {
        (
            self.encoding_table[(b as usize) >> 4],
            self.encoding_table[(b as usize) & 0xF],
        )
    }
}

impl Encoder for HexEncoder {
    fn encoded_len(&self, data: &[u8]) -> Result<usize, Error> {
        data.len().checked_mul(2).ok_or(Error::IntegerOverflow)
    }

    fn encode_slice(&self, data: &[u8], target: &mut [u8]) -> Result<usize, Error> {
        let encoded_len: usize = self.encoded_len(data)?;
        if target.len() < encoded_len {
            Err(Error::InsufficientTargetSpace)
        } else {
            let target: &mut [u8] = &mut target[..encoded_len];
            for (d, t) in data.iter().zip(target.chunks_exact_mut(2)) {
                let (a, b) = self.encode(*d);
                t[0] = a;
                t[1] = b;
            }
            Ok(encoded_len)
        }
    }
}

impl StringEncoder for HexEncoder {
    fn encode_string(&self, data: &[u8], target: &mut String) -> Result<usize, Error> {
        unsafe { encode_string_unchecked(self, data, target) }
    }
}
