use std::marker::PhantomData;

use crate::{Decoder, Error};

/// Responsible for decoding hex data.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct HexDecoder {
    _phantom: PhantomData<()>,
}

impl Default for HexDecoder {
    fn default() -> Self {
        Self {
            _phantom: PhantomData::default(),
        }
    }
}

impl HexDecoder {
    //! Decoding Table

    /// The invalid byte.
    pub(in crate::hex) const INV: u8 = 0xFF;

    /// The decoding table.
    pub(in crate::hex) const DECODING_TABLE: [u8; 256] = Self::create_decoding_table();

    /// Creates the decoding table.
    const fn create_decoding_table() -> [u8; 256] {
        let mut decoding_table: [u8; 256] = [Self::INV; 256];
        decoding_table[b'0' as usize] = 0x00;
        decoding_table[b'1' as usize] = 0x01;
        decoding_table[b'2' as usize] = 0x02;
        decoding_table[b'3' as usize] = 0x03;
        decoding_table[b'4' as usize] = 0x04;
        decoding_table[b'5' as usize] = 0x05;
        decoding_table[b'6' as usize] = 0x06;
        decoding_table[b'7' as usize] = 0x07;
        decoding_table[b'8' as usize] = 0x08;
        decoding_table[b'9' as usize] = 0x09;
        decoding_table[b'a' as usize] = 0x0A;
        decoding_table[b'b' as usize] = 0x0B;
        decoding_table[b'c' as usize] = 0x0C;
        decoding_table[b'd' as usize] = 0x0D;
        decoding_table[b'e' as usize] = 0x0E;
        decoding_table[b'f' as usize] = 0x0F;
        decoding_table[b'A' as usize] = 0x0A;
        decoding_table[b'B' as usize] = 0x0B;
        decoding_table[b'C' as usize] = 0x0C;
        decoding_table[b'D' as usize] = 0x0D;
        decoding_table[b'E' as usize] = 0x0E;
        decoding_table[b'F' as usize] = 0x0F;
        decoding_table
    }
}

impl HexDecoder {
    //! Decoding

    /// Decodes the high and low hex bytes.
    #[inline(always)]
    pub fn decode(high: u8, low: u8) -> u8 {
        (Self::DECODING_TABLE[high as usize] << 4) | (Self::DECODING_TABLE[low as usize])
    }
}

impl Decoder for HexDecoder {
    fn decoded_len(&self, data: &[u8]) -> Result<usize, Error> {
        if data.len() % 2 != 0 {
            Err(Error::InvalidEncodedData)
        } else {
            Ok(data.len() / 2)
        }
    }

    fn decode_slice(&self, data: &[u8], target: &mut [u8]) -> Result<usize, Error> {
        let decoded_len: usize = self.decoded_len(data)?;
        if target.len() < decoded_len {
            Err(Error::InsufficientTargetSpace)
        } else {
            let target: &mut [u8] = &mut target[..decoded_len];
            for (d, t) in data.chunks_exact(2).zip(target.iter_mut()) {
                *t = Self::decode(d[0], d[1])
            }
            Ok(decoded_len)
        }
    }
}
