use crate::Error;

/// Responsible for encoding data.
pub trait Encoder {
    /// Gets the length of the encoded data.
    fn encoded_len(&self, data: &[u8]) -> Result<usize, Error>;

    /// Encodes the data into the target slice. Returns the length of the encoded data.
    fn encode_slice(&self, data: &[u8], target: &mut [u8]) -> Result<usize, Error>;

    /// Encodes the data into the target vec. Returns the length of the encoded data.
    fn encode_vec(&self, data: &[u8], target: &mut Vec<u8>) -> Result<usize, Error> {
        let encoded_len: usize = self.encoded_len(data)?;
        let old_len: usize = target.len();
        let new_len: usize = old_len
            .checked_add(encoded_len)
            .ok_or(Error::IntegerOverflow)?;
        target.resize(new_len, 0u8);
        let slice: &mut [u8] = &mut target.as_mut_slice()[old_len..];
        match self.encode_slice(data, slice) {
            Ok(encoded_len) => Ok(encoded_len),
            Err(error) => {
                target.truncate(old_len);
                Err(error)
            }
        }
    }

    /// Encodes the data as a vec. Returns the vec.
    fn encode_as_vec(&self, data: &[u8]) -> Result<Vec<u8>, Error> {
        let mut target: Vec<u8> = Vec::new();
        self.encode_vec(data, &mut target)?;
        Ok(target)
    }
}

/// Responsible for encoding data as UTF-8 strings.
pub trait StringEncoder: Encoder {
    /// Encodes the data into the target string. Returns the length of the encoded data.
    fn encode_string(&self, data: &[u8], target: &mut String) -> Result<usize, Error>;

    /// Encodes the data as a string. Returns the string.
    fn encode_as_string(&self, data: &[u8]) -> Result<String, Error> {
        let mut target: String = String::new();
        self.encode_string(data, &mut target)?;
        Ok(target)
    }
}

/// Encodes the data into the target string with the given encoder. Returns the length of the encoded data.
/// This function does not check if the encoded data is valid UTF-8.
pub(in crate) unsafe fn encode_string_unchecked<E: Encoder>(
    encoder: &E,
    data: &[u8],
    target: &mut String,
) -> Result<usize, Error> {
    let encoded_len: usize = encoder.encoded_len(data)?;
    let old_len: usize = target.len();
    let new_len: usize = old_len
        .checked_add(encoded_len)
        .ok_or(Error::IntegerOverflow)?;
    let vec: &mut Vec<u8> = target.as_mut_vec();
    vec.resize(new_len, 0u8);
    let slice: &mut [u8] = &mut vec.as_mut_slice()[old_len..];
    match encoder.encode_slice(data, slice) {
        Ok(encoded_len) => Ok(encoded_len),
        Err(error) => {
            vec.truncate(old_len);
            Err(error)
        }
    }
}
