use crate::Error;

/// Responsible for decoding data.
pub trait Decoder {
    /// Gets the length of the decoded data.
    fn decoded_len(&self, data: &[u8]) -> Result<usize, Error>;

    /// Decodes the data into the target slice. Returns the length of the decoded data.
    fn decode_slice(&self, data: &[u8], target: &mut [u8]) -> Result<usize, Error>;

    /// Decodes the data into the target vec. Returns the length of the decoded data.
    fn decode_vec(&self, data: &[u8], target: &mut Vec<u8>) -> Result<usize, Error> {
        let decoded_len: usize = self.decoded_len(data)?;
        let old_len: usize = target.len();
        let new_len: usize = old_len
            .checked_add(decoded_len)
            .ok_or(Error::IntegerOverflow)?;
        target.resize(new_len, 0u8);
        let slice: &mut [u8] = &mut target.as_mut_slice()[old_len..];
        match self.decode_slice(data, slice) {
            Ok(decoded_len) => Ok(decoded_len),
            Err(error) => {
                target.truncate(old_len);
                Err(error)
            }
        }
    }

    /// Decodes the data as a vec. Returns the vec.
    fn decode_as_vec(&self, data: &[u8]) -> Result<Vec<u8>, Error> {
        let mut target: Vec<u8> = Vec::new();
        self.decode_vec(data, &mut target)?;
        Ok(target)
    }
}
