# Emver

This module was designed to address the problem of releasing updates to Embassy Packages where the upstream project was
either unaware of or apathetic towards supporting their application on the Embassy platform. In most cases, the original
package will support [semver2](https://semver.org/spec/v2.0.0.html). This leaves us with the problem where we would like
to preserve the original package's version, since one of the goals of the Embassy platform is transparency. However, on
occasion, we have screwed up and published a version of a package that needed to have its metadata updated. In this
scenario we were left with the conundrum of either unilaterally claiming a version number of a package we did not author
or let the issue persist until the next update. Neither of these promote good user experiences, for different reasons.
This module extends the semver standard linked above with a 4th digit, which is given PATCH semantics.

## Usage

Add this to your `Cargo.toml`

```toml
[dependencies]
emver = "0.1.0"
```

## Operations

A `Version` contains 4 components: major, minor, patch, and revision. The meaning of first three can be found in the
Semver2 specification. The fourth is also given patch semantics but is intended to be incremented by package
distributors when they are not themselves authors.

A `Version` can also be parsed from a dot separated string like `0.1.2.3`. They can also be serialized to strings, but
in cases where the last digit is zero, the last dot and the zero are omitted. Parsing will still work over a triple. The
relevant parse function for `Version` is `parse_version` and it is generated by the `nom` parser library. It can be
applied to a string and will produce a `Result<Version, ParseError>`.

The other half of this library deals with the type `VersionRange`. A `VersionRange` is a set that is either anchored at
a particular `Version` with some sort of comparison operator: `= >= <= > <` or it is described as a conjunction or
disjunction of other `VersionRange`s. For convenience we also provide two constructors (`Any`, `None`) to serve as
identity elements on the `Conj` and `Disj` constructors respectively. As a result, to gain maximum performance, you
should use the `conj` and `disj` smart constructors as opposed to their dumb counterparts `Conj` and `Disj`. This will
immediately evaluate the identities and annihilators as opposed to building up the AST further, saving peak memory.

For convenience, there are two Monoid wrappers exposed: (`AnyRange`, `AllRange`). This allows you to `fold` an `Iterable`
with the `combine` operation seeded with the `empty` value. The semantic differences are whether or not `combine` uses
`disj` or `conj` respectively.

Most of the time you will want to parse these values from strings, but the internals are exposed for the rarer cases.
Some of the grammar from `node-semver` is supported (^1.2.3, ~2.3.4, 1.2.x, 0.0.0-1.1.1) as well.

Finally, the most useful operation in this package is the `satisfies` operation on `Version` with the argument of a
`VersionRange`. This is simply a predicate that tells you whether the `Version` falls inside the `VersionRange`.

## Laws
All laws listed below are equality of observation, not a literal `Eq` instance giving representational Equality. The
only observer that this library has is the `satisfies` operation. When you read "a === b", you should interpret that as
obs.satisfies(a) === obs.satisfies(b). These laws simply mean that it is always safe to do a substitution of a term on
the LHS for a term on the RHS without changing the meaning of your program.

- `Conj` is commutative: conj(a,b) === conj(b,a)
- `Disj` is commutative: conj(a,b) === disj(b,a)
- `Conj` is associative: conj(conj(a,b),c) === conj(a,conj(b,c))
- `Disj` is associative: disj(disj(a,b),c) === disj(a,disj(b,c))
- `Any` is identity of `Conj`: conj(a, Any) === a
- `None` is identity of `Disj`: disj(a, None) === a
- `Any` annihilates `Disj`: disj(a, Any) === Any
- `None` annihilates `Conj`: conj(a, None) === None
- `Conj` distributes over `Disj`: conj(a,disj(b,c)) === disj(conj(a,b),conj(a,c))
- `Disj` distributes over `Conj`: disj(a,conj(b,c)) === conj(disj(a,b),disj(a,c))
