//! Abstraction for hardware timers/clocks

use crate::{
    duration::Duration, fixed_point::FixedPoint, fraction::Fraction, instant::Instant,
    time_int::TimeInt, timer::param, timer::Timer,
};
use core::hash::Hash;

/// Potential `Clock` errors
#[non_exhaustive]
#[derive(Debug, Eq, PartialEq, Hash)]
pub enum Error {
    /// Exact cause of failure is unknown
    Unspecified,
    /// The clock has either stopped or never started
    NotRunning,
}

impl Default for Error {
    fn default() -> Self {
        Self::Unspecified
    }
}

/// The `Clock` trait provides an abstraction for hardware-specific timer peripherals, external
/// timer devices, RTCs, etc.
///
/// The `Clock` is characterized by an inner unsigned integer storage type (either [`u32`] or
/// [`u64`]), a [`u32`]/[`u32`] [`Fraction`] defining the duration (in seconds) of one
/// count of the `Clock`, and a custom error type representing errors that may be generated by the
/// implementation.
///
/// In addition to the [`Clock::try_now()`] method which returns an [`Instant`],
/// software [`Timer`]s can be spawned from a `Clock` object.
pub trait Clock: Sized {
    /// The type to hold the tick count
    type T: TimeInt + Hash;

    /// The duration of one clock tick in seconds, AKA the clock precision.
    const SCALING_FACTOR: Fraction;

    /// Get the current Instant
    ///
    /// # Errors
    ///
    /// - [`Error::NotRunning`]
    /// - [`Error::Unspecified`]
    fn try_now(&self) -> Result<Instant<Self>, Error>;

    /// Spawn a new, `OneShot` [`Timer`] from this clock
    fn new_timer<Dur: Duration>(
        &self,
        duration: Dur,
    ) -> Timer<param::OneShot, param::Armed, Self, Dur>
    where
        Dur: FixedPoint,
    {
        Timer::<param::None, param::None, Self, Dur>::new(&self, duration)
    }
}
