use embedded_graphics::{
    geometry::Point,
    mock_display::MockDisplay,
    mono_font::{ascii::FONT_6X9, MonoTextStyle},
    pixelcolor::BinaryColor,
    primitives::Rectangle,
    Drawable,
};

use crate::{
    alignment::HorizontalAlignment,
    rendering::test::{assert_rendered, assert_styled_rendered},
    style::{TextBoxStyle, TextBoxStyleBuilder},
    utils::test::size_for,
    TextBox,
};

#[test]
fn simple_render() {
    assert_rendered(
        HorizontalAlignment::Right,
        "word",
        size_for(&FONT_6X9, 6, 1),
        &[
            "            ........................",
            "            ......................#.",
            "            ......................#.",
            "            #...#...##...#.#....###.",
            "            #.#.#..#..#..##.#..#..#.",
            "            #.#.#..#..#..#.....#..#.",
            "            .#.#....##...#......###.",
            "            ........................",
            "            ........................",
        ],
    );
}

#[test]
fn simple_render_cr() {
    let mut display = MockDisplay::new();
    display.set_allow_overdraw(true);

    let character_style = MonoTextStyle::new(&FONT_6X9, BinaryColor::On);
    let style = TextBoxStyle::with_alignment(HorizontalAlignment::Right);

    TextBox::with_textbox_style(
        "O\rX",
        Rectangle::new(Point::zero(), size_for(&FONT_6X9, 3, 1)),
        character_style,
        style,
    )
    .draw(&mut display)
    .unwrap();

    display.assert_pattern(&[
        "                  ",
        "            ##### ",
        "            ## ## ",
        "            # # # ",
        "            # # # ",
        "            ## ## ",
        "            ##### ",
    ]);
}

#[test]
fn simple_word_wrapping() {
    assert_rendered(
        HorizontalAlignment::Right,
        "word wrapping",
        size_for(&FONT_6X9, 9, 2),
        &[
            "                              ........................",
            "                              ......................#.",
            "                              ......................#.",
            "                              #...#...##...#.#....###.",
            "                              #.#.#..#..#..##.#..#..#.",
            "                              #.#.#..#..#..#.....#..#.",
            "                              .#.#....##...#......###.",
            "                              ........................",
            "                              ........................",
            "      ................................................",
            "      ................................#...............",
            "      ................................................",
            "      #...#..#.#....###..###...###...##....###....##..",
            "      #.#.#..##.#..#..#..#..#..#..#...#....#..#..#..#.",
            "      #.#.#..#.....#..#..#..#..#..#...#....#..#..#..#.",
            "      .#.#...#......###..###...###...###...#..#...###.",
            "      ...................#.....#....................#.",
            "      ...................#.....#..................##..",
        ],
    );
}

#[test]
fn simple_word_wrapping_with_spaces() {
    assert_styled_rendered(
        TextBoxStyleBuilder::new()
            .alignment(HorizontalAlignment::Right)
            .leading_spaces(true)
            .trailing_spaces(true)
            .build(),
        " word wrapping",
        size_for(&FONT_6X9, 9, 2),
        &[
            "                  ....................................",
            "                  ............................#.......",
            "                  ............................#.......",
            "                  ......#...#...##...#.#....###.......",
            "                  ......#.#.#..#..#..##.#..#..#.......",
            "                  ......#.#.#..#..#..#.....#..#.......",
            "                  .......#.#....##...#......###.......",
            "                  ....................................",
            "                  ....................................",
            "      ................................................",
            "      ................................#...............",
            "      ................................................",
            "      #...#..#.#....###..###...###...##....###....##..",
            "      #.#.#..##.#..#..#..#..#..#..#...#....#..#..#..#.",
            "      #.#.#..#.....#..#..#..#..#..#...#....#..#..#..#.",
            "      .#.#...#......###..###...###...###...#..#...###.",
            "      ...................#.....#....................#.",
            "      ...................#.....#..................##..",
        ],
    );
}

#[test]
fn word_longer_than_line_wraps_word() {
    assert_rendered(
        HorizontalAlignment::Right,
        "word  somereallylongword",
        size_for(&FONT_6X9, 9, 3),
        &[
            "                              ........................",
            "                              ......................#.",
            "                              ......................#.",
            "                              #...#...##...#.#....###.",
            "                              #.#.#..#..#..##.#..#..#.",
            "                              #.#.#..#..#..#.....#..#.",
            "                              .#.#....##...#......###.",
            "                              ........................",
            "                              ........................",
            "......................................................",
            "...........................................##....##...",
            "............................................#.....#...",
            "..###...##..##.#....##...#.#....##....###...#.....#...",
            ".##....#..#.#.#.#..#.##..##.#..#.##..#..#...#.....#...",
            "...##..#..#.#.#.#..##....#.....##....#..#...#.....#...",
            ".###....##..#...#...###..#......###...###..###...###..",
            "......................................................",
            "......................................................",
            "......................................................",
            ".......##...........................................#.",
            "........#...........................................#.",
            ".#..#...#.....##...###....##..#...#...##...#.#....###.",
            ".#..#...#....#..#..#..#..#..#.#.#.#..#..#..##.#..#..#.",
            ".#..#...#....#..#..#..#..#..#.#.#.#..#..#..#.....#..#.",
            "..###..###....##...#..#...###..#.#....##...#......###.",
            ".#..#.......................#.........................",
            "..##......................##..........................",
        ],
    );
}

#[test]
fn first_word_longer_than_line_wraps_word() {
    assert_rendered(
        HorizontalAlignment::Right,
        "somereallylongword",
        size_for(&FONT_6X9, 9, 2),
        &[
            "......................................................",
            "...........................................##....##...",
            "............................................#.....#...",
            "..###...##..##.#....##...#.#....##....###...#.....#...",
            ".##....#..#.#.#.#..#.##..##.#..#.##..#..#...#.....#...",
            "...##..#..#.#.#.#..##....#.....##....#..#...#.....#...",
            ".###....##..#...#...###..#......###...###..###...###..",
            "......................................................",
            "......................................................",
            "......................................................",
            ".......##...........................................#.",
            "........#...........................................#.",
            ".#..#...#.....##...###....##..#...#...##...#.#....###.",
            ".#..#...#....#..#..#..#..#..#.#.#.#..#..#..##.#..#..#.",
            ".#..#...#....#..#..#..#..#..#.#.#.#..#..#..#.....#..#.",
            "..###..###....##...#..#...###..#.#....##...#......###.",
            ".#..#.......................#.........................",
            "..##......................##..........................",
        ],
    );
}

#[test]
fn soft_hyphen_rendering() {
    assert_rendered(
        HorizontalAlignment::Right,
        "soft\u{AD}hyphen",
        size_for(&FONT_6X9, 6, 2),
        &[
            "      ..............................",
            "      ...............#....#.........",
            "      ..............#.#...#.........",
            "      ..###...##....#....###........",
            "      .##....#..#..###....#...#####.",
            "      ...##..#..#...#.....#.#.......",
            "      .###....##....#......#........",
            "      ..............................",
            "      ..............................",
            "....................................",
            ".#.................#................",
            ".#.................#................",
            ".###...#..#..###...###....##...###..",
            ".#..#..#..#..#..#..#..#..#.##..#..#.",
            ".#..#..#..#..#..#..#..#..##....#..#.",
            ".#..#...###..###...#..#...###..#..#.",
            ".......#..#..#......................",
            "........##...#......................",
        ],
    );
}
