use std::env;
use std::fmt::{Debug, Display, Formatter};
use std::fs::{self, File};
use std::io::{self, BufWriter, ErrorKind, Seek, stdout, Write};
use std::mem::size_of;
use std::path::{Path, PathBuf};

use object::{Architecture, BinaryFormat, Endianness, LittleEndian, pod, RelocationEncoding, RelocationKind, SectionKind, SymbolFlags, SymbolKind, SymbolScope, U16, U32};
use object::pe::{IMAGE_RESOURCE_DATA_IS_DIRECTORY, ImageResourceDataEntry, ImageResourceDirectory, ImageResourceDirectoryEntry};
use object::write::{self, Object, Relocation, Symbol, SymbolSection};

use crate::manifest::ManifestBuilder;

#[cfg(test)]
mod test;

/// Embeds the manifest described by `manifest` by converting it to XML,
/// then saving it to a file and passing the correct options to the linker
/// on MSVC targets, or by building a static library and instructing Cargo
/// to link the executable against it on GNU targets.
pub fn embed_manifest(manifest: ManifestBuilder) -> Result<(), Error> {
    let out_dir = get_out_dir()?;
    let target = get_target()?;
    if matches!(target.os, TargetOs::WindowsMsvc) {
        let manifest_file = out_dir.join("manifest.xml");
        write!(BufWriter::new(File::create(&manifest_file)?), "{}", manifest)?;
        link_manifest_msvc(&manifest_file, &mut stdout().lock())
    } else {
        let manifest_data = manifest.to_string();
        link_manifest_gnu(manifest_data.as_bytes(), &out_dir, target.arch, &mut stdout().lock())
    }
}

/// Directly embeds the manifest in the provided `file` by passing the correct
/// options to the linker on MSVC targets, or by building a static library
/// and instructing Cargo to link the executable against it on GNU targets.
pub fn embed_manifest_file<P: AsRef<Path>>(file: P) -> Result<(), io::Error> {
    let out_dir = get_out_dir()?;
    let target = get_target()?;
    if matches!(target.os, TargetOs::WindowsMsvc) {
        Ok(link_manifest_msvc(file.as_ref(), &mut stdout().lock())?)
    } else {
        let manifest = fs::read(file.as_ref())?;
        Ok(link_manifest_gnu(&manifest, &out_dir, target.arch, &mut stdout().lock())?)
    }
}

fn get_out_dir() -> Result<PathBuf, io::Error> {
    match env::var_os("OUT_DIR") {
        Some(dir) => Ok(PathBuf::from(dir)),
        None => env::current_dir()
    }
}

enum TargetOs { WindowsGnu, WindowsMsvc }

struct Target { arch: Architecture, os: TargetOs }

fn get_target() -> Result<Target, Error> {
    match env::var("TARGET") {
        Ok(target) => parse_target(&target),
        _ => Err(Error { repr: Repr::UnknownTarget })
    }
}

fn parse_target(target: &str) -> Result<Target, Error> {
    let mut iter = target.splitn(3, '-');
    let arch = match iter.next() {
        Some("i686") => Architecture::I386,
        Some("aarch64") => Architecture::Aarch64,
        Some("x86_64") => Architecture::X86_64,
        _ => return Err(Error { repr: Repr::UnknownTarget })
    };
    if iter.next() != Some("pc") {
        return Err(Error { repr: Repr::UnknownTarget })
    }
    let os = match iter.next() {
        Some("windows-gnu") => TargetOs::WindowsGnu,
        Some("windows-msvc") => TargetOs::WindowsMsvc,
        _ => return Err(Error { repr: Repr::UnknownTarget })
    };
    Ok(Target { arch, os })
}

fn link_manifest_msvc<W: Write>(manifest_path: &Path, out: &mut W) -> Result<(), Error> {
    writeln!(out, "cargo:rustc-link-arg-bins=/MANIFEST:EMBED")?;
    writeln!(out, "cargo:rustc-link-arg-bins=/MANIFESTINPUT:{}", manifest_path.canonicalize()?.display())?;
    writeln!(out, "cargo:rustc-link-arg-bins=/MANIFESTUAC:NO")?;
    Ok(())
}

fn link_manifest_gnu<W: Write>(manifest: &[u8], out_dir: &Path, arch: Architecture, out: &mut W) -> Result<(), Error> {
    // Generate a COFF object file containing the manifest in a .rsrc section.
    let object_file = create_object_file(manifest, arch);
    let object_data = match object_file.write() {
        Ok(data) => data,
        Err(e) => return Err(Error { repr: Repr::ObjectError(e) })
    };

    // Save the object file to a library, asking Cargo to link against it.
    fs::create_dir_all(out_dir)?;
    writeln!(out, "cargo:rustc-link-search=native={}", out_dir.display())?;
    let path = out_dir.join("libembed-manifest.a");
    writeln!(out, "cargo:rustc-link-lib=static=embed-manifest")?;
    let mut file = BufWriter::new(File::create(&path)?);
    create_library(&mut file, &object_data)?;
    Ok(())
}

fn create_object_file(manifest: &[u8], arch: Architecture) -> Object {
    // Define an object file with two sections to be combined by the linker.
    let mut obj = Object::new(BinaryFormat::Coff, arch, Endianness::Little);
    let rsrc01 = obj.add_section(Vec::new(), b".rsrc$01".to_vec(), SectionKind::Data);
    let rsrc02 = obj.add_section(Vec::new(), b".rsrc$02".to_vec(), SectionKind::Data);

    // Add symbols for the manifest data and sections.
    let symbol = obj.add_symbol(Symbol {
        name: b"$R000000".to_vec(), value: 0, size: 0, kind: SymbolKind::Label, scope: SymbolScope::Compilation,
        weak: false, section: SymbolSection::Section(rsrc02), flags: SymbolFlags::None
    });
    obj.section_symbol(rsrc01);
    obj.section_symbol(rsrc02);

    // Create resource directories for type ID 24, name ID 1, language ID 1033.
    let mut buf: Vec<u8> = Vec::with_capacity(100);
    for (id, subdir) in [(24u32, true), (1, true), (1033, false)] {
        let dir = ImageResourceDirectory {
            characteristics: Default::default(), time_date_stamp: Default::default(),
            major_version: Default::default(), minor_version: Default::default(),
            number_of_named_entries: Default::default(), number_of_id_entries: U16::new(LittleEndian, 1)
        };
        buf.extend_from_slice(pod::bytes_of(&dir));
        let mut addr = (buf.len() + size_of::<ImageResourceDirectoryEntry>()) as u32;
        if subdir { addr |= IMAGE_RESOURCE_DATA_IS_DIRECTORY; }
        let entry = ImageResourceDirectoryEntry {
            name_or_id: U32::new(LittleEndian, id),
            offset_to_data_or_directory: U32::new(LittleEndian, addr)
        };
        buf.extend_from_slice(pod::bytes_of(&entry));
    }
    obj.append_section_data(rsrc01, &buf, 4);

    // Add resource data entry with relocation of its address to `symbol`.
    let data_entry = ImageResourceDataEntry {
        offset_to_data: Default::default(), size: U32::new(LittleEndian, manifest.len() as u32),
        code_page: Default::default(), reserved: Default::default()
    };
    let offset = obj.append_section_data(rsrc01, pod::bytes_of(&data_entry), 4);
    obj.add_relocation(rsrc01, Relocation {
        offset, size: 32, kind: RelocationKind::ImageOffset, encoding: RelocationEncoding::Generic, symbol, addend: 0
    }).expect("invalid relocation");

    // Add the manifest data to the resource data section.
    obj.add_symbol_data(symbol, rsrc02, manifest, 4);

    // Use the populated object file.
    obj
}

fn create_library<W: Write + Seek>(file: &mut W, object_file: &[u8]) -> io::Result<()> {
    // Archive file signature.
    file.write_all(b"!<arch>\n")?;

    // Empty linker member (only one for GNU targets).
    writeln!(file, "{:16}{:<12}{:<6}{:<6}{:<8o}{:<10}\x60", "/", 0, 0, 0, 0, 4)?;
    file.write_all(&0u32.to_le_bytes())?;

    // Manifest object file header.
    writeln!(file, "{:16}{:<12}{:<6}{:<6}{:<8o}{:<10}\x60", "manifest.o", 0, 0, 0, 0o100644, object_file.len())?;
    file.write_all(object_file)?;
    file.flush()
}

#[derive(Debug)]
pub struct Error {
    repr: Repr
}

#[derive(Debug)]
enum Repr {
    IoError(io::Error),
    ObjectError(write::Error),
    UnknownTarget
}

impl Display for Error {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self.repr {
            Repr::IoError(ref e) => write!(f, "I/O error: {}", e),
            Repr::ObjectError(ref e) => write!(f, "object: {}", e),
            Repr::UnknownTarget => f.write_str("unknown target")
        }
    }
}

impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        match self.repr {
            Repr::IoError(ref e) => Some(e),
            _ => None
        }
    }
}

impl From<io::Error> for Error {
    fn from(e: io::Error) -> Self {
        Error { repr: Repr::IoError(e) }
    }
}

impl From<Error> for io::Error {
    fn from(e: Error) -> Self {
        match e.repr {
            Repr::IoError(ioe) => ioe,
            _ => io::Error::new(ErrorKind::Other, e)
        }
    }
}
