//! The `embed-manifest` crate provides a straightforward way to embed
//! a Windows manifest in an executable, whatever the build environment,
//! without dependencies on external tools from LLVM or MinGW.
//!
//! This should be called from a [build script][1], as shown below.
//!
//! [1]: https://doc.rust-lang.org/cargo/reference/build-scripts.html
//!
//! On MSVC targets, the manifest file is embedded in the executable by
//! instructing Cargo to pass `/MANIFEST` options to `LINK.EXE`. This
//! requires Cargo from Rust 1.56.
//!
//! On GNU targets, the manifest file is added as a resource in a static
//! library file, and Cargo is instructed to link this file into the
//! executable.
//!
//! # Usage
//!
//! This crate should be added to the `[build-dependencies]` section in
//! your executable’s `Cargo.toml`:
//!
//! ```toml
//! [build-dependencies]
//! embed-manifest = "1"
//! ```
//!
//! In the same directory, create a `build.rs` file to call this crate’s
//! code, and to only be run when the manifest is changed:
//!
//! ```
//! use embed_manifest::embed_manifest_file;
//!
//! fn main() {
//!     # let tempdir = tempfile::tempdir().unwrap();
//!     # std::env::set_var("OUT_DIR", tempdir.path());
//!     # std::env::set_var("TARGET", "x86_64-pc-windows-gnu");
//!     embed_manifest_file("sample.exe.manifest")
//!         .expect("unable to embed manifest file");
//!     println!("cargo:rerun-if-changed=sample.exe.manifest");
//! }
//! ```
//!
//! And create a manifest file with a matching name, like `sample.exe.manifest`,
//! with the right configuration for your executable:
//!
//! ```xml
//! <?xml version="1.0" encoding="UTF-8" standalone="yes"?>
//! <assembly xmlns="urn:schemas-microsoft-com:asm.v1" xmlns:asmv3="urn:schemas-microsoft-com:asm.v3" manifestVersion="1.0">
//!     <assemblyIdentity name="Sample.Test" type="win32" version="0.0.0.0"/>
//!     <dependency>
//!         <dependentAssembly>
//!             <assemblyIdentity type="win32" name="Microsoft.Windows.Common-Controls" version="6.0.0.0" processorArchitecture="*" publicKeyToken="6595b64144ccf1df" language="*"/>
//!         </dependentAssembly>
//!     </dependency>
//!     <asmv3:application>
//!        <asmv3:windowsSettings>
//!             <activeCodePage xmlns="http://schemas.microsoft.com/SMI/2019/WindowsSettings">UTF-8</activeCodePage>
//!             <dpiAware xmlns="http://schemas.microsoft.com/SMI/2005/WindowsSettings">True/PM</dpiAware>
//!             <dpiAwareness xmlns="http://schemas.microsoft.com/SMI/2016/WindowsSettings">PerMonitorV2, PerMonitor</dpiAwareness>
//!             <heapType xmlns="http://schemas.microsoft.com/SMI/2020/WindowsSettings">SegmentHeap</heapType>
//!             <longPathAware xmlns="http://schemas.microsoft.com/SMI/2016/WindowsSettings">true</longPathAware>
//!         </asmv3:windowsSettings>
//!     </asmv3:application>
//!     <asmv3:trustInfo>
//!         <asmv3:security>
//!             <asmv3:requestedPrivileges>
//!                 <asmv3:requestedExecutionLevel level="asInvoker" uiAccess="false"/>
//!             </asmv3:requestedPrivileges>
//!         </asmv3:security>
//!     </asmv3:trustInfo>
//! </assembly>
//! ```

#![allow(clippy::needless_doctest_main)]

pub use embed::embed_manifest_file;

mod embed;
