use std::fs;
use std::path::{Path, PathBuf};

use object::{Architecture, LittleEndian, Object, ObjectSection, ObjectSymbol, pod, RelocationEncoding, RelocationKind, SectionKind};
use object::coff::CoffFile;
use object::pe::{IMAGE_RESOURCE_DATA_IS_DIRECTORY, ImageResourceDataEntry, ImageResourceDirectory, ImageResourceDirectoryEntry};
use object::read::archive::{ArchiveFile, ArchiveKind};
use tempfile::{TempDir, tempdir};

use crate::embed::{create_object_file, embed_file};

const MANIFEST: &[u8] = br#"<assembly xmlns="urn:schemas-microsoft-com:asm.v1" xmlns:asmv3="urn:schemas-microsoft-com:asm.v3" manifestVersion="1.0"/>"#;

#[test]
fn create_lib() {
    let res = do_embed_file("x86_64-pc-windows-gnu");
    let data = fs::read(&res.lib()).unwrap();
    let archive = ArchiveFile::parse(&data[..]).unwrap();
    assert_eq!(archive.kind(), ArchiveKind::Gnu);
    assert_eq!(archive.members().map(|m| m.unwrap().name()).collect::<Vec<&[u8]>>(), &[b"manifest.o"]);
}

#[test]
fn link_lib_gnu() {
    let res = do_embed_file("x86_64-pc-windows-gnu");
    assert!(res.lib().exists());
    let mut search_option = String::from("cargo:rustc-link-search=native=");
    search_option.push_str(res.out_dir.path().to_str().unwrap());
    assert_eq!(res.lines(), &[search_option.as_str(), "cargo:rustc-link-lib=static=embed-manifest"]);
}

#[test]
fn link_manifest_msvc() {
    let res = do_embed_file("x86_64-pc-windows-msvc");
    assert!(!res.lib().exists());
    let mut input_option = String::from("cargo:rustc-link-arg-bins=/MANIFESTINPUT:");
    input_option.push_str(res.manifest_path.canonicalize().unwrap().to_str().unwrap());
    assert_eq!(res.lines(), &[
        "cargo:rustc-link-arg-bins=/MANIFEST:EMBED",
        input_option.as_str(),
        "cargo:rustc-link-arg-bins=/MANIFESTUAC:NO"
    ]);
}

struct EmbedResult {
    manifest_path: PathBuf,
    out_dir: TempDir,
    output: String
}

impl EmbedResult {
    fn lib(&self) -> PathBuf {
        self.out_dir.path().join("libembed-manifest.a")
    }

    fn lines(&self) -> Vec<&str> {
        self.output.lines().collect()
    }
}

fn do_embed_file(target: &str) -> EmbedResult {
    let manifest_path = Path::new(env!("CARGO_MANIFEST_DIR")).join("sample.exe.manifest");
    let out_dir = tempdir().unwrap();
    let mut buf: Vec<u8> = Vec::new();
    embed_file(&manifest_path, out_dir.path(), target, &mut buf).unwrap();
    EmbedResult { manifest_path, out_dir, output: String::from_utf8(buf).unwrap() }
}

#[test]
fn object_file_x86() {
    let file = create_object_file(MANIFEST, Architecture::I386).write().unwrap();
    let obj = CoffFile::parse(&file[..]).unwrap();
    assert_eq!(obj.architecture(), Architecture::I386);
    check_object_file(obj);
}

#[test]
fn object_file_x86_64() {
    let file = create_object_file(MANIFEST, Architecture::X86_64).write().unwrap();
    let obj = CoffFile::parse(&file[..]).unwrap();
    assert_eq!(obj.architecture(), Architecture::X86_64);
    check_object_file(obj);
}

#[test]
#[ignore]
fn object_file_aarch64() {
    let file = create_object_file(MANIFEST, Architecture::Aarch64).write().unwrap();
    let obj = CoffFile::parse(&file[..]).unwrap();
    assert_eq!(obj.architecture(), Architecture::Aarch64);
    check_object_file(obj);
}

fn check_object_file(obj: CoffFile) {
    // There should be two sections `.rsrc$01` and `.rsrc$02.
    assert_eq!(obj.sections().map(|s| s.name().unwrap().to_string()).collect::<Vec<_>>(), &[".rsrc$01", ".rsrc$02"]);

    // There should be two section symbols and one for relocation.
    assert_eq!(obj.symbols().map(|s| s.name().unwrap().to_string()).collect::<Vec<_>>(), &["$R000000", ".rsrc$01", ".rsrc$02"]);

    // The resource sections must be data sections.
    let rsrc01 = obj.section_by_name(".rsrc$01").unwrap();
    let rsrc02 = obj.section_by_name(".rsrc$02").unwrap();
    assert_eq!(rsrc01.address(), 0);
    assert_eq!(rsrc01.kind(), SectionKind::Data);
    assert_eq!(rsrc02.address(), 0);
    assert_eq!(rsrc02.kind(), SectionKind::Data);

    // The data RVA in the resource data entry must be relocatable.
    let (addr, reloc) = rsrc01.relocations().next().unwrap();
    assert_eq!(reloc.kind(), RelocationKind::ImageOffset);
    assert_eq!(reloc.encoding(), RelocationEncoding::Generic);
    assert_eq!(addr, 0x48); // size of the directory table, three directories, and no strings
    assert_eq!(reloc.addend(), 0);

    // The resource directory contains one manifest resource type subdirectory.
    let data = rsrc01.data().unwrap();
    let (dir, rest) = pod::from_bytes::<ImageResourceDirectory>(data).unwrap();
    assert_eq!(0, dir.number_of_named_entries.get(LittleEndian));
    assert_eq!(1, dir.number_of_id_entries.get(LittleEndian));
    let (entries, _) = pod::slice_from_bytes::<ImageResourceDirectoryEntry>(rest, 1).unwrap();
    assert_eq!(24, entries[0].name_or_id.get(LittleEndian));
    let offset = entries[0].offset_to_data_or_directory.get(LittleEndian);
    assert_eq!(IMAGE_RESOURCE_DATA_IS_DIRECTORY, offset & IMAGE_RESOURCE_DATA_IS_DIRECTORY);
    let offset = (offset & !IMAGE_RESOURCE_DATA_IS_DIRECTORY) as usize;

    // The manifest subdirectory contains one image (not DLL) manifest subdirectory.
    let (dir, rest) = pod::from_bytes::<ImageResourceDirectory>(&data[offset..]).unwrap();
    assert_eq!(0, dir.number_of_named_entries.get(LittleEndian));
    assert_eq!(1, dir.number_of_id_entries.get(LittleEndian));
    let (entries, _) = pod::slice_from_bytes::<ImageResourceDirectoryEntry>(rest, 1).unwrap();
    assert_eq!(1, entries[0].name_or_id.get(LittleEndian));
    let offset = entries[0].offset_to_data_or_directory.get(LittleEndian);
    assert_eq!(IMAGE_RESOURCE_DATA_IS_DIRECTORY, offset & IMAGE_RESOURCE_DATA_IS_DIRECTORY);
    let offset = (offset & !IMAGE_RESOURCE_DATA_IS_DIRECTORY) as usize;

    // The image manifest subdirectory contains one US English manifest data entry.
    let (dir, rest) = pod::from_bytes::<ImageResourceDirectory>(&data[offset..]).unwrap();
    assert_eq!(0, dir.number_of_named_entries.get(LittleEndian));
    assert_eq!(1, dir.number_of_id_entries.get(LittleEndian));
    let (entries, _) = pod::slice_from_bytes::<ImageResourceDirectoryEntry>(rest, 1).unwrap();
    assert_eq!(0x0409, entries[0].name_or_id.get(LittleEndian));
    let offset = entries[0].offset_to_data_or_directory.get(LittleEndian);
    assert_eq!(0, offset & IMAGE_RESOURCE_DATA_IS_DIRECTORY);
    let offset = offset as usize;

    // The manifest data matches what was added.
    let (entry, _) = pod::from_bytes::<ImageResourceDataEntry>(&data[offset..]).unwrap();
    let end = entry.size.get(LittleEndian) as usize;
    let resource_data = rsrc02.data().unwrap();
    let manifest = &resource_data[..end];
    assert_eq!(MANIFEST, manifest);
}
