//! The `embed-manifest` crate provides a straightforward way to embed
//! a Windows manifest in an executable, whatever the build environment,
//! without dependencies on external tools from LLVM or MinGW.
//!
//! This should be called from a [build script][1], as shown below.
//!
//! [1]: https://doc.rust-lang.org/cargo/reference/build-scripts.html
//!
//! On MSVC targets, the manifest file is embedded in the executable by
//! instructing Cargo to pass `/MANIFEST` options to `LINK.EXE`. This
//! requires Cargo from Rust 1.56.
//!
//! On GNU targets, the manifest file is added as a resource in a static
//! library file, and Cargo is instructed to link this file into the
//! executable.
//!
//! # Usage
//!
//! This crate should be added to the `[build-dependencies]` section in
//! your executable’s `Cargo.toml`:
//!
//! ```toml
//! [build-dependencies]
//! embed-manifest = "1"
//! ```
//!
//! In the same directory, create a `build.rs` file to call this crate’s
//! code, and to only be run when the manifest is changed:
//!
//! ```
//! use embed_manifest::embed_manifest_file;
//!
//! fn main() {
//!     # std::env::set_var("TARGET", "x86_64-pc-windows-gnu");
//!     # std::env::set_var("OUT_DIR", "target/tmp");
//!     embed_manifest_file("sample.exe.manifest")
//!         .expect("unable to embed manifest file");
//!     println!("cargo:rerun-if-changed=sample.exe.manifest");
//! }
//! ```
//!
//! And create a manifest file with a matching name, like `sample.exe.manifest`,
//! with the right configuration for your executable:
//!
//! ```xml
//! <?xml version="1.0" encoding="UTF-8" standalone="yes"?>
//! <assembly xmlns="urn:schemas-microsoft-com:asm.v1" xmlns:asmv3="urn:schemas-microsoft-com:asm.v3" manifestVersion="1.0">
//!     <assemblyIdentity name="Sample.Test" type="win32" version="0.0.0.0"/>
//!     <dependency>
//!         <dependentAssembly>
//!             <assemblyIdentity type="win32" name="Microsoft.Windows.Common-Controls" version="6.0.0.0" processorArchitecture="*" publicKeyToken="6595b64144ccf1df" language="*"/>
//!         </dependentAssembly>
//!     </dependency>
//!     <asmv3:application>
//!        <asmv3:windowsSettings>
//!             <activeCodePage xmlns="http://schemas.microsoft.com/SMI/2019/WindowsSettings">UTF-8</activeCodePage>
//!             <dpiAware xmlns="http://schemas.microsoft.com/SMI/2005/WindowsSettings">True/PM</dpiAware>
//!             <dpiAwareness xmlns="http://schemas.microsoft.com/SMI/2016/WindowsSettings">PerMonitorV2, PerMonitor</dpiAwareness>
//!             <heapType xmlns="http://schemas.microsoft.com/SMI/2020/WindowsSettings">SegmentHeap</heapType>
//!             <longPathAware xmlns="http://schemas.microsoft.com/SMI/2016/WindowsSettings">true</longPathAware>
//!         </asmv3:windowsSettings>
//!     </asmv3:application>
//!     <asmv3:trustInfo>
//!         <asmv3:security>
//!             <asmv3:requestedPrivileges>
//!                 <asmv3:requestedExecutionLevel level="asInvoker" uiAccess="false"/>
//!             </asmv3:requestedPrivileges>
//!         </asmv3:security>
//!     </asmv3:trustInfo>
//! </assembly>
//! ```

use std::{env, fs, io};
use std::fs::File;
use std::io::{BufWriter, Write};
use std::path::{Path, PathBuf};

/// Directly embeds the manifest in the provided `file` by passing the correct
/// options to the linker on MSVC targets, or by building a static library
/// and instructing Cargo to link the executable against it on GNU targets.
pub fn embed_manifest_file<P: AsRef<Path>>(file: P) -> io::Result<()> {
    embed_file(file.as_ref())
}

fn embed_file(path: &Path) -> io::Result<()> {
    // Check for an MSVC target where the linker can embed the manifest directly.
    let target = env::var("TARGET").unwrap_or_else(|_e| String::new());
    if target.ends_with("msvc") {
        println!("cargo:rustc-link-arg-bins=/MANIFEST:EMBED");
        println!("cargo:rustc-link-arg-bins=/MANIFESTINPUT:{}", path.canonicalize()?.display());
        println!("cargo:rustc-link-arg-bins=/MANIFESTUAC:NO");
        return Ok(())
    }

    // Generate a COFF object file containing the manifest in a .rsrc section.
    // Since this is a single section containing a single user-defined resource,
    // we can take some shortcuts.
    let manifest = fs::read(path)?;
    let resources = create_resources(&manifest);
    let object_file = create_object_file(&resources, &target);

    // Save the object file to a library, asking Cargo to link against it.
    let mut path = match env::var_os("OUT_DIR") {
        Some(out_dir) => PathBuf::from(out_dir),
        None => env::current_dir()?
    };
    fs::create_dir_all(&path)?;
    println!("cargo:rustc-link-search=native={}", path.display());
    path.push("libembed-manifest.a");
    println!("cargo:rustc-link-lib=static=embed-manifest");
    let mut file = BufWriter::new(File::create(&path)?);
    create_library(&mut file, &object_file)
}

fn padded_len(buf: &[u8], align: usize) -> usize {
    let len = buf.len();
    let padding = align - (len % align);
    if padding == align { len } else { len + padding }
}

fn create_resources(manifest: &[u8]) -> Vec<u8> {
    // Add entries for type ID 24, name ID 1, language ID 1033.
    let mut buf: Vec<u8> = Vec::with_capacity(manifest.len() + 100);
    for (id, subdir) in [(24u32, true), (1, true), (1033, false)] {
        // No flags, timestamp, version, or name entries, one ID entry.
        buf.extend_from_slice(&[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0]);
        buf.extend_from_slice(&id.to_le_bytes());
        let mut addr = (buf.len() + 4) as u32;
        if subdir { addr |= 0x80000000; }
        buf.extend_from_slice(&addr.to_le_bytes());
    }

    // Add resource data entry and data.
    buf.extend_from_slice(&((buf.len() + 16) as u32).to_le_bytes());
    buf.extend_from_slice(&(manifest.len() as u32).to_le_bytes());
    buf.extend_from_slice(&[0, 0, 0, 0, 0, 0, 0, 0]);
    buf.extend_from_slice(manifest);

    // Pad and return the resources.
    buf.resize(padded_len(&buf, 8), 0);
    buf
}

fn create_object_file(resource: &[u8], target: &str) -> Vec<u8> {
    // Sizes of parts of the file.
    const HEADER_LEN: u32 = 60;
    let resource_len = padded_len(resource, 4) as u32;
    const RELOCATIONS_LEN: u32 = 10;
    const SYMBOL_TABLE_LEN: u32 = 18;

    // COFF file header.
    let machine_type: u16 = if target.starts_with("x86_64") {
        0x8664
    } else if target.starts_with("aarch64") {
        0xaa64
    } else {
        0x014c
    };
    let number_of_sections: u16 = 1;
    let timestamp: u32 = 0;
    let pointer_to_symbol_table: u32 = HEADER_LEN + resource_len + RELOCATIONS_LEN;
    let number_of_symbols: u32 = 1;
    let optional_header_size: u16 = 0;
    let characteristics: u16 = if machine_type == 0x014c { 0x104 } else { 4 };

    // COFF section header for .rsrc section.
    let section_name = b".rsrc\0\0\0";
    let virtual_size: u32 = 0;
    let virtual_address: u32 = 0;
    let size_of_raw_data: u32 = resource.len() as u32;
    let pointer_to_raw_data: u32 = HEADER_LEN;
    let pointer_to_relocations: u32 = HEADER_LEN + resource_len;
    let pointer_to_line_numbers: u32 = 0;
    let number_of_relocations: u16 = 1;
    let number_of_line_numbers: u16 = 0;
    let section_characteristics: u32 = 0xc0300040;

    // Relocation for .rsrc section.
    let reloc_virtual_address: u32 = 0x48;
    let reloc_symbol_table_index: u32 = 0;
    let reloc_type: u16 = match machine_type {
        0x8664 => 3, // IMAGE_REL_AMD64_ADDR32NB
        0xaa64 => 2, // IMAGE_REL_ARM64_ADDR32NB
        _ => 7       // IMAGE_REL_I386_DIR32NB
    };

    // Symbol table for .rsrc section.
    let s_name = section_name;
    let s_value: u32 = 0;
    let s_section_number: i16 = 1;
    let s_type: u16 = 0;
    let s_storage_class: u8 = 3;
    let s_aux_symbols: u8 = 0;
    let string_table_length: u32 = 4;

    // Return the formatted object file.
    let mut buf: Vec<u8> = Vec::with_capacity((HEADER_LEN + resource_len + RELOCATIONS_LEN + SYMBOL_TABLE_LEN) as usize);
    buf.extend_from_slice(&machine_type.to_le_bytes());
    buf.extend_from_slice(&number_of_sections.to_le_bytes());
    buf.extend_from_slice(&timestamp.to_le_bytes());
    buf.extend_from_slice(&pointer_to_symbol_table.to_le_bytes());
    buf.extend_from_slice(&number_of_symbols.to_le_bytes());
    buf.extend_from_slice(&optional_header_size.to_le_bytes());
    buf.extend_from_slice(&characteristics.to_le_bytes());
    buf.extend_from_slice(section_name);
    buf.extend_from_slice(&virtual_size.to_le_bytes());
    buf.extend_from_slice(&virtual_address.to_le_bytes());
    buf.extend_from_slice(&size_of_raw_data.to_le_bytes());
    buf.extend_from_slice(&pointer_to_raw_data.to_le_bytes());
    buf.extend_from_slice(&pointer_to_relocations.to_le_bytes());
    buf.extend_from_slice(&pointer_to_line_numbers.to_le_bytes());
    buf.extend_from_slice(&number_of_relocations.to_le_bytes());
    buf.extend_from_slice(&number_of_line_numbers.to_le_bytes());
    buf.extend_from_slice(&section_characteristics.to_le_bytes());
    buf.extend_from_slice(resource);
    buf.resize((HEADER_LEN + resource_len) as usize, 0);
    buf.extend_from_slice(&reloc_virtual_address.to_le_bytes());
    buf.extend_from_slice(&reloc_symbol_table_index.to_le_bytes());
    buf.extend_from_slice(&reloc_type.to_le_bytes());
    buf.extend_from_slice(s_name);
    buf.extend_from_slice(&s_value.to_le_bytes());
    buf.extend_from_slice(&s_section_number.to_le_bytes());
    buf.extend_from_slice(&s_type.to_le_bytes());
    buf.extend_from_slice(&s_storage_class.to_le_bytes());
    buf.extend_from_slice(&s_aux_symbols.to_le_bytes());
    buf.extend_from_slice(&string_table_length.to_le_bytes());
    buf
}

fn create_library<W: Write>(file: &mut W, object_file: &[u8]) -> io::Result<()> {
    // Archive file signature.
    file.write_all(b"!<arch>\n")?;

    // Empty linker member.
    writeln!(file, "{:16}{:<12}{:<6}{:<6}{:<8o}{:<10}\x60", "/", 0, 0, 0, 0, 4)?;
    file.write_all(&[0, 0, 0, 0])?;

    // Manifest object file header.
    writeln!(file, "{:16}{:<12}{:<6}{:<6}{:<8o}{:<10}\x60", "manifest.o", 0, 0, 0, 0o100644, object_file.len())?;
    file.write_all(object_file)?;
    file.flush()
}
