//! PSET coinjoin example
//! 1. Person `A` create a transcation with 1 input and 3 outputs(1 fee output)
//! 2. Person `B` takes the transcation from A and adds one input and two outputs
//!    which transact another confidential asset
//! 3. Person `B` blinds it's own outputs and gives the pset back to A
//! 4. B completly blinds the transaction
//! 5. B signs the blinded Transaction and sends it back to A
//! 6. A signs it's input
//! 7. A finalizes the pset
//! 8. A extracts and broadcasts the transaction
//! During the entire interaction, the output blinding factors for A and B are not
//! shared with each other.
extern crate bitcoin;
extern crate elements;
extern crate rand;
extern crate serde_json;

use std::{collections::HashMap, str::FromStr};

use elements::confidential::{AssetBlindingFactor, ValueBlindingFactor};
use elements::{
    bitcoin::PublicKey, pset::PartiallySignedTransaction as Pset, OutPoint,
    Script, TxOutSecrets, TxOutWitness, Txid, WScriptHash,
};
use elements::{pset, secp256k1_zkp};

use elements::encode::{deserialize, serialize_hex};
use elements::hashes::hex::FromHex;
use elements::{confidential, AssetId, TxOut};
use rand::SeedableRng;

// Assume txouts are simple pay to wpkh
// and keep the secrets correponding to
// confidential txouts
#[derive(Debug, Clone)]
struct Secrets {
    sk: bitcoin::PrivateKey,
    sec: TxOutSecrets,
}

fn parse_txout(txout_info: &str) -> (TxOut, Secrets, pset::Input) {
    // Parse the string of data into serde_json::Value.
    let v: serde_json::Value = serde_json::from_str(txout_info).unwrap();

    let txout = TxOut {
        asset: deserialize::<confidential::Asset>(
            &Vec::<u8>::from_hex(&v["assetcommitment"].as_str().unwrap()).unwrap(),
        )
        .unwrap(),
        value: deserialize::<confidential::Value>(
            &Vec::<u8>::from_hex(&v["amountcommitment"].as_str().unwrap()).unwrap(),
        )
        .unwrap(),
        nonce: deserialize::<confidential::Nonce>(
            &Vec::<u8>::from_hex(&v["commitmentnonce"].as_str().unwrap()).unwrap(),
        )
        .unwrap(),
        script_pubkey: Script::from_hex(&v["scriptPubKey"].as_str().unwrap()).unwrap(),
        witness: TxOutWitness::default(),
    };

    let txoutsecrets = Secrets {
        sk: bitcoin::PrivateKey::from_wif(&v["skwif"].as_str().unwrap()).unwrap(),
        sec: TxOutSecrets {
            asset_bf: AssetBlindingFactor::from_str(&v["assetblinder"].as_str().unwrap()).unwrap(),
            value_bf: ValueBlindingFactor::from_str(&v["amountblinder"].as_str().unwrap()).unwrap(),
            value: bitcoin::Amount::from_str_in(
                &v["amount"].as_str().unwrap(),
                bitcoin::Denomination::Bitcoin,
            )
            .unwrap()
            .as_sat(),
            asset: AssetId::from_hex(&v["asset"].as_str().unwrap()).unwrap(),
        },
    };

    let inp = pset::Input::from_prevout(OutPoint::new(
        Txid::from_str(&v["txid"].as_str().unwrap()).unwrap(),
        v["vout"].as_u64().unwrap() as u32,
    ));

    (txout, txoutsecrets, inp)
}

fn txout_data() -> [(TxOut, Secrets, pset::Input); 2] {
    // Some JSON input data as a &str. Maybe this comes from the user.
    let asset_txout = r#"
    {
        "txid": "55855ab698631f8dc9c11aaa299fbd62f05869b082ecb14395372a6c6e95ff7a",
        "vout": 1,
        "scriptPubKey": "0014011d384302576b408aa3686db874e2b17cc2b01b",
        "amount": "10.00000000",
        "assetcommitment": "0ac55f449ddb6853f2508766d5afb9f3b45e41a8ef5368cad75fb88e5e249395d1",
        "asset": "4fa41f2929d4bf6975a55967d9da5b650b6b9bfddeae4d7b54b04394be328f7f",
        "amountcommitment": "097d88c92ca814a207f73441c56cee943f0bb2556da194c14a4b912b078c2238ae",
        "amountblinder": "bcfa96f8068c91cf7b80c197066d1fc0d756606bf6666c9f78e120a653b7d13e",
        "assetblinder": "f4ba5cf033c0557bbaab295a057a87c943f3639a05a62bef37f29dc18aa45886",
        "commitmentnonce": "025341cb5e4e2d8cb69e694cb20e5ea4cc8ddf2801180096fd071addfcd8bc4445",
        "skwif": "cU52mfNAru457o7DQmmb1TpkNasXmg63QLPH1F94LEZSzJe2uK3V"
    }"#;

    let btc_txout = r#"
    {
        "txid": "70478b6898407362d43e9e56fd72a89b0556ac2593ed6e025c16376bba315180",
        "vout": 0,
        "scriptPubKey": "0014d2cbec8783bd01c9f178348b08500a830a89a7f9",
        "amount": "2.30000000",
        "assetcommitment": "0bb9325c276764451bbc2eb82a4c8c4bb6f4007ba803e5a5ba72d0cd7c09848e1a",
        "asset": "b2e15d0d7a0c94e4e2ce0fe6e8691b9e451377f6e46e8045a86f7c4b5d4f0f23",
        "amountcommitment": "091622d935953bf06e0b7393239c68c6f810a00fe19d11c6ae343cffd3037077da",
        "amountblinder": "0f155ac96c49e39c0501e3448e9aac89f5b43c16bf9156e6c1694e310c80f374",
        "assetblinder": "de6ecd62ab6fc66597b2144f38c3be873ba583970aacdfcc8978a1a0b6cb872c",
        "commitmentnonce": "02535fe4ad0fcd675cd0f62bf73b60a554dc1569b80f1f76a2bbfc9f00d439bf4b",
        "skwif": "cRrq6NyygXvNBHW7ozK6b33F1qZqbNbKTVtTSQph947jgPKN8WCH"
      }"#;

    [parse_txout(btc_txout), parse_txout(asset_txout)]
}

fn test_data() -> HashMap<String, String> {
    let mut tests = HashMap::new();
    tests.insert(
        String::from("empty"),
        String::from("70736574ff01020402000000010401000105010001fb040200000000"),
    );
    tests.insert(String::from("A_pset_unblinded") , String::from("70736574ff01020402000000010401010105010301fb04020000000001017a0bb9325c276764451bbc2eb82a4c8c4bb6f4007ba803e5a5ba72d0cd7c09848e1a091622d935953bf06e0b7393239c68c6f810a00fe19d11c6ae343cffd3037077da02535fe4ad0fcd675cd0f62bf73b60a554dc1569b80f1f76a2bbfc9f00d439bf4b160014d2cbec8783bd01c9f178348b08500a830a89a7f9010e20805131ba6b37165c026eed9325ac56059ba872fd569e3ed462734098688b4770010f04000000000001030820a107000000000007fc04707365740220230f4f5d4b7c6fa845806ee4f67713459e1b69e8e60fcee2e4940c7a0d5de1b20104220020e5793ad956ee91ebf3543b37d110701118ed4078ffa0d477eacb8885e486ad8507fc047073657406210212bf0ea45b733dfde8ecb5e896306c4165c666c99fc5d1ab887f71393a975cea07fc047073657407210212bf0ea45b733dfde8ecb5e896306c4165c666c99fc5d1ab887f71393a975cea07fc0470736574080400000000000103086ce2ad0d0000000007fc04707365740220230f4f5d4b7c6fa845806ee4f67713459e1b69e8e60fcee2e4940c7a0d5de1b20104220020f6b43d56e004e9d0b1ec2fc3c95511d81af08420992be8dec7f86cdf8970b3e207fc04707365740621027d07ae478c0aa607321643cb5e8ed59ee1f5ff4d9d55efedec066ccb1f5d537d07fc04707365740721027d07ae478c0aa607321643cb5e8ed59ee1f5ff4d9d55efedec066ccb1f5d537d07fc047073657408040000000000010308f40100000000000007fc04707365740220230f4f5d4b7c6fa845806ee4f67713459e1b69e8e60fcee2e4940c7a0d5de1b201040000"));
    tests.insert(String::from("pset_coinjoined_unblinded") , String::from("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"));
    tests.insert(String::from("pset_coinjoined_B_blinded") , String::from("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"));
    tests.insert(String::from("pset_coinjoined_blinded") , String::from("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"));
    // tests.insert(String::from("blinded_one_inp_signed") , String::from("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"));
    // tests.insert(String::from("blinded_signed") , String::from("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"));
    // tests.insert(String::from("finalized") , String::from("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"));
    // tests.insert(String::from("extracted_tx") , String::from("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"));

    tests
}

fn main() {
    let tests = test_data();
    // Initially secp context and rng global state
    let secp = secp256k1_zkp::Secp256k1::new();
    #[allow(deprecated)]
    let mut rng = rand::ChaChaRng::seed_from_u64(0);

    let txouts = txout_data();
    let (btc_txout, btc_txout_secrets, btc_inp) = txouts[0].clone();
    let (asset_txout, asset_txout_secrets, asset_inp) = txouts[1].clone();

    let mut pset = Pset::new_v2();
    assert_eq!(serialize_hex(&pset), tests["empty"]);

    // Add the btc asset input
    let mut btc_inp = btc_inp;
    btc_inp.witness_utxo = Some(btc_txout.clone());
    pset.add_input(btc_inp);

    // Create the first txout
    let dest_btc_wsh =
        WScriptHash::from_str("e5793ad956ee91ebf3543b37d110701118ed4078ffa0d477eacb8885e486ad85")
            .unwrap();
    let dest_btc_amt = 500_000; // sat
    let dest_btc_blind_pk =
        PublicKey::from_str("0212bf0ea45b733dfde8ecb5e896306c4165c666c99fc5d1ab887f71393a975cea")
            .unwrap();
    let dest_btc_txout = TxOut {
        asset: confidential::Asset::Explicit(btc_txout_secrets.sec.asset),
        value: confidential::Value::Explicit(dest_btc_amt),
        nonce: confidential::Nonce::Confidential(dest_btc_blind_pk.key),
        script_pubkey: Script::new_v0_wsh(&dest_btc_wsh),
        witness: TxOutWitness::default(),
    };

    // Create the change txout
    let btc_fees_amt = 500; // sat
    let change_amt = btc_txout_secrets.sec.value - dest_btc_amt - btc_fees_amt;
    let change_btc_blind_pk =
        PublicKey::from_str("027d07ae478c0aa607321643cb5e8ed59ee1f5ff4d9d55efedec066ccb1f5d537d")
            .unwrap();
    let change_btc_wsh =
        WScriptHash::from_str("f6b43d56e004e9d0b1ec2fc3c95511d81af08420992be8dec7f86cdf8970b3e2")
            .unwrap();
    let change_btc_txout = TxOut {
        asset: confidential::Asset::Explicit(btc_txout_secrets.sec.asset),
        value: confidential::Value::Explicit(change_amt),
        nonce: confidential::Nonce::Confidential(change_btc_blind_pk.key),
        script_pubkey: Script::new_v0_wsh(&change_btc_wsh),
        witness: TxOutWitness::default(),
    };

    // Create the fee txout
    let btc_fees_txout = TxOut::new_fee(btc_fees_amt, btc_txout_secrets.sec.asset);

    pset.add_output(pset::Output::from_txout(dest_btc_txout));
    pset.add_output(pset::Output::from_txout(change_btc_txout));
    pset.add_output(pset::Output::from_txout(btc_fees_txout));

    // Mark owned outputs for blinding later
    // This tells that person that controls input zero is responsible for
    // blinding outputs 0, 1, 2.
    pset.outputs[0].blinding_key = Some(dest_btc_blind_pk);
    pset.outputs[0].blinder_index = Some(0);
    pset.outputs[1].blinding_key = Some(change_btc_blind_pk);
    pset.outputs[1].blinder_index = Some(0);

    // pset after adding the information about the bitcoin input from A
    // Pset with 2 input and 3 outputs
    assert_eq!(serialize_hex(&pset), tests["A_pset_unblinded"]);
    // ----------------------------------------------------------
    // Party A sends unblinded pset to B. Step 1 completed

    // Add the asset input
    let mut asset_inp = asset_inp;
    asset_inp.witness_utxo = Some(asset_txout.clone());
    pset.add_input(asset_inp);

    // Add outputs
    // Send 5_000 worth of asset units to new address
    // Create the first asset txout(fourth output)
    let dest_asset_wsh =
        WScriptHash::from_str("37831b3ee29fc96f8e61ccb98fbe2dcb03e189dd29cfecc691b5a7442d8548e8")
            .unwrap();
    let dest_asset_amt = 1_000_000; // sat
    let dest_asset_blind_pk =
        PublicKey::from_str("03d559d2a5a4180f418a69c4bed5508971cda9313722fff71e053d3d82fee9d7bd")
            .unwrap();
    let dest_asset_txout = TxOut {
        asset: confidential::Asset::Explicit(asset_txout_secrets.sec.asset),
        value: confidential::Value::Explicit(dest_asset_amt),
        nonce: confidential::Nonce::Confidential(dest_asset_blind_pk.key),
        script_pubkey: Script::new_v0_wsh(&dest_asset_wsh),
        witness: TxOutWitness::default(),
    };

    // Create the change txout
    let change_asset_amt = asset_txout_secrets.sec.value - dest_asset_amt;
    let change_asset_blind_pk =
        PublicKey::from_str("029e5980b4f9b9a9fd568c1c4b48631a800c310405ae8b2ac41ddaf87add3062f1")
            .unwrap();
    let change_asset_wsh =
        WScriptHash::from_str("e7da55d19cc85b0420c539a90b667d4d85f59ee0ed417493a947c3a2256cc0aa")
            .unwrap();
    let change_asset_txout = TxOut {
        asset: confidential::Asset::Explicit(asset_txout_secrets.sec.asset),
        value: confidential::Value::Explicit(change_asset_amt),
        nonce: confidential::Nonce::Confidential(change_asset_blind_pk.key),
        script_pubkey: Script::new_v0_wsh(&change_asset_wsh),
        witness: TxOutWitness::default(),
    };

    // Add the outputs
    pset.add_output(pset::Output::from_txout(dest_asset_txout));
    pset.add_output(pset::Output::from_txout(change_asset_txout));

    // This tells that person that controls input index one is responsible for
    // blinding outputs 3, 4.
    pset.outputs[3].blinding_key = Some(dest_asset_blind_pk);
    pset.outputs[3].blinder_index = Some(1);
    pset.outputs[4].blinding_key = Some(change_asset_blind_pk);
    pset.outputs[4].blinder_index = Some(1);

    // pset after adding the information about the bitcoin input from A
    // and adding B's input. Two inputs and 5 outputs
    assert_eq!(serialize_hex(&pset), tests["pset_coinjoined_unblinded"]);
    // ----------------------------------------------------------
    // B Adds it's own outputs. Step 2 completed
    // ----- Step 3: B to blind it's own outputs
    let inp_txout_sec = [
        None,
        Some(&asset_txout_secrets.sec),
    ];
    pset.blind_non_last(&mut rng, &secp, &inp_txout_sec).unwrap();
    assert_eq!(serialize_hex(&pset), tests["pset_coinjoined_B_blinded"]);

    // Step 4: A blinds it's own inputs
    let inp_txout_sec = [
        Some(&btc_txout_secrets.sec),
        None,
    ];
    pset.blind_last(&mut rng, &secp, &inp_txout_sec).unwrap();
    assert_eq!(serialize_hex(&pset), tests["pset_coinjoined_blinded"]);

    // check whether the blinding was correct
    // Verify the balance checks
    let tx = pset.extract_tx().unwrap();
    // println!("{}", serialize_hex(&tx));
    tx.verify_tx_amt_proofs(&secp, &[btc_txout, asset_txout])
        .unwrap();
}
