use anyhow::Result;
use electrify::{ConnectionBuilder, Ipv4Net};
use jacklog::{debug, instrument};
use structopt::{clap::AppSettings::ColoredHelp, StructOpt};

/// Electrify automates management of peers in a WireGuard network, using the
/// wireguard-tools CLI and SSH access.
#[derive(Clone, Debug, PartialEq, PartialOrd, StructOpt)]
#[structopt(setting = ColoredHelp)]
struct Config {
    /// Increase log level.
    #[structopt(short, long, parse(from_occurrences))]
    verbose: usize,

    #[structopt(subcommand)]
    subcommand: SubCommand,
}

#[derive(Clone, Debug, PartialEq, PartialOrd, StructOpt)]
#[structopt(setting = ColoredHelp)]
enum SubCommand {
    Peer(PeerSubCommand),
    Connection(ConnectionSubCommand),
}

/// Various peer-management operations.
#[derive(Clone, Debug, PartialEq, PartialOrd, StructOpt)]
#[structopt(setting = ColoredHelp)]
enum PeerSubCommand {
    Add(PeerAdd),
}

/// Various connection-management operations.
///
/// This is usually the place to start when adding a node to the network.
#[derive(Clone, Debug, PartialEq, PartialOrd, StructOpt)]
#[structopt(setting = ColoredHelp)]
enum ConnectionSubCommand {
    New(ConnectionNew),
}

/// Add a peer to an existing connection.
#[derive(Clone, Debug, PartialEq, PartialOrd, StructOpt)]
#[structopt(setting = ColoredHelp)]
struct PeerAdd {
    #[structopt(short, long)]
    address: String,
}

/// Add a peer to an existing connection.
#[derive(Clone, Debug, PartialEq, PartialOrd, StructOpt)]
#[structopt(setting = ColoredHelp)]
struct ConnectionNew {
    #[structopt(short, long)]
    interface: String,

    #[structopt(short, long)]
    peer_ipv4: Ipv4Net,

    #[structopt(long, visible_alias = "ip")]
    ipv4: Ipv4Net,
}

#[instrument]
fn main() -> Result<()> {
    let config = Config::from_args();
    jacklog::from_level(config.verbose + 1, Some(&[env!("CARGO_BIN_NAME")]))?;
    debug!(?config);

    match config.subcommand {
        SubCommand::Peer(p) => match p {
            PeerSubCommand::Add(_) => {}
        },
        SubCommand::Connection(c) => match c {
            ConnectionSubCommand::New(ConnectionNew {
                interface,
                peer_ipv4,
                ipv4,
            }) => {
                let conn = ConnectionBuilder::default()
                    .interface(interface)
                    .peer_ipv4(peer_ipv4)
                    .ipv4(ipv4)
                    .build()?;
                debug!(?conn);

                conn.setup()?;
            }
        },
    };

    Ok(())
}
