use crate::error::{PacketError, PacketProtocolError};
use byteorder::{ByteOrder, LittleEndian};
use derive_more::{Display, From, Into, UpperHex};
use electricui_embedded::prelude::*;
use std::{fmt, str};

#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Into)]
pub struct OwnedMessageId(Vec<u8>);

impl OwnedMessageId {
    pub fn new(id: &[u8]) -> Option<Self> {
        MessageId::new(id).map(|id| Self::from_wire(&id))
    }

    pub fn as_bytes(&self) -> &[u8] {
        &self.0
    }

    pub fn as_str(&self) -> Result<&str, str::Utf8Error> {
        str::from_utf8(&self.0)
    }

    pub fn from_utf8(s: &str) -> Self {
        Self(s.as_bytes().to_vec())
    }

    #[allow(clippy::len_without_is_empty)]
    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn as_wire(&self) -> MessageId<'_> {
        unsafe { MessageId::new_unchecked(&self.0) }
    }

    pub fn from_wire(id: &MessageId<'_>) -> Self {
        Self(id.as_bytes().to_vec())
    }
}

impl fmt::Display for OwnedMessageId {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Ok(s) = self.as_str() {
            f.write_str(s)
        } else {
            write!(f, "{:X?}", self.0)
        }
    }
}

#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Display)]
#[display(fmt = "Id({}), Type({}), Data({:02X?})", id, typ, data)]
pub struct Variable {
    pub id: OwnedMessageId,
    pub typ: MessageType,
    pub data: Vec<u8>,
}

// TODO zero is invalid, no From
#[derive(
    Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Display, UpperHex, From, Into,
)]
pub struct BoardId(u16);

impl BoardId {
    pub fn encode_request<T: AsRef<[u8]> + AsMut<[u8]>>(
        p: &mut Packet<T>,
    ) -> Result<(), PacketError> {
        p.set_data_length(0)?;
        p.set_typ(MessageType::U16);
        p.set_internal(true);
        p.set_offset(false);
        p.set_id_length(MessageId::INTERNAL_BOARD_ID.len() as _)?;
        p.set_response(true);
        p.set_acknum(0);
        p.msg_id_mut()?
            .copy_from_slice(MessageId::INTERNAL_BOARD_ID.as_bytes());
        p.set_checksum(p.compute_checksum()?)?;
        Ok(())
    }

    pub fn decode_response<T: AsRef<[u8]>>(p: &Packet<T>) -> Result<Self, PacketProtocolError> {
        // TODO - sanity check protocol
        let id = p.payload()?;
        Ok(LittleEndian::read_u16(id).into())
    }
}

// TODO - empty is invalid
#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, From, Into)]
pub struct BoardName(Vec<u8>);

impl BoardName {
    pub fn as_str(&self) -> Result<&str, str::Utf8Error> {
        str::from_utf8(&self.0)
    }

    #[allow(clippy::len_without_is_empty)]
    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn encode_request<T: AsRef<[u8]> + AsMut<[u8]>>(
        p: &mut Packet<T>,
    ) -> Result<(), PacketError> {
        p.set_data_length(0)?;
        p.set_typ(MessageType::Callback);
        p.set_internal(false);
        p.set_offset(false);
        p.set_id_length(MessageId::BOARD_NAME.len() as _)?;
        p.set_response(true);
        p.set_acknum(0);
        p.msg_id_mut()?
            .copy_from_slice(MessageId::BOARD_NAME.as_bytes());
        p.set_checksum(p.compute_checksum()?)?;
        Ok(())
    }

    pub fn decode_response<T: AsRef<[u8]>>(p: &Packet<T>) -> Result<Self, PacketProtocolError> {
        // TODO - sanity check protocol
        let name = p.payload()?;
        Ok(name.to_vec().into())
    }
}

impl fmt::Display for BoardName {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Ok(s) = self.as_str() {
            f.write_str(s)
        } else {
            write!(f, "{:X?}", self.0)
        }
    }
}

#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug)]
pub struct WritableIdsAnnouncement;

impl WritableIdsAnnouncement {
    pub fn encode_request<T: AsRef<[u8]> + AsMut<[u8]>>(
        p: &mut Packet<T>,
    ) -> Result<(), PacketError> {
        p.set_data_length(0)?;
        p.set_typ(MessageType::Callback);
        p.set_internal(true);
        p.set_offset(false);
        p.set_id_length(1)?;
        p.set_response(true);
        p.set_acknum(0);
        p.msg_id_mut()?
            .copy_from_slice(MessageId::INTERNAL_AM.as_bytes());
        p.set_checksum(p.compute_checksum()?)?;
        Ok(())
    }
}

// TODO - allow accumulating
#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, From, Into)]
pub struct IdsAnnouncement(Vec<OwnedMessageId>);

impl IdsAnnouncement {
    pub fn as_slice(&self) -> &[OwnedMessageId] {
        &self.0
    }

    #[allow(clippy::len_without_is_empty)]
    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn decode_response<T: AsRef<[u8]>>(p: &Packet<T>) -> Result<Self, PacketProtocolError> {
        // TODO - sanity check protocol
        let ids: Vec<OwnedMessageId> = p
            .payload()?
            .split(|b| *b == b'\0')
            .filter_map(OwnedMessageId::new)
            .collect();
        Ok(Self(ids))
    }
}

#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Into)]
pub struct WritableIdsAnnouncementEndList(usize);

impl WritableIdsAnnouncementEndList {
    pub fn decode_response<T: AsRef<[u8]>>(p: &Packet<T>) -> Result<Self, PacketProtocolError> {
        // TODO - sanity check protocol
        let num_ids = p.payload()?;
        Ok(Self(if p.typ() == MessageType::U8 {
            num_ids[0] as usize
        } else {
            // u16
            LittleEndian::read_u16(num_ids) as usize
        }))
    }
}

#[derive(Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Default, Into)]
pub struct TrackedVariables(Vec<Variable>);

impl TrackedVariables {
    pub fn as_slice(&self) -> &[Variable] {
        &self.0
    }

    #[allow(clippy::len_without_is_empty)]
    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn encode_request<T: AsRef<[u8]> + AsMut<[u8]>>(
        p: &mut Packet<T>,
    ) -> Result<(), PacketError> {
        p.set_data_length(0)?;
        p.set_typ(MessageType::Callback);
        p.set_internal(true);
        p.set_offset(false);
        p.set_id_length(MessageId::INTERNAL_AV.len() as _)?;
        p.set_response(true);
        p.set_acknum(0);
        p.msg_id_mut()?
            .copy_from_slice(MessageId::INTERNAL_AV.as_bytes());
        p.set_checksum(p.compute_checksum()?)?;
        Ok(())
    }

    // TODO - accumulating here or alt pattern
    pub fn decode_response_accumulating<T: AsRef<[u8]>>(
        &mut self,
        p: &Packet<T>,
    ) -> Result<(), PacketProtocolError> {
        // TODO - sanity check protocol
        let id = p.msg_id()?;
        let typ = p.typ();
        let data = p.payload()?;
        self.0.push(Variable {
            id: OwnedMessageId::from_wire(&id),
            typ,
            data: data.to_vec(),
        });
        Ok(())
    }
}

#[derive(
    Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash, Debug, Display, UpperHex, From, Into,
)]
pub struct Heartbeat(u8);

impl Heartbeat {
    pub fn encode_request<T: AsRef<[u8]> + AsMut<[u8]>>(
        &self,
        p: &mut Packet<T>,
    ) -> Result<(), PacketError> {
        p.set_data_length(1)?;
        p.set_typ(MessageType::U8);
        p.set_internal(true);
        p.set_offset(false);
        p.set_id_length(MessageId::INTERNAL_HEARTBEAT.len() as _)?;
        p.set_response(true);
        p.set_acknum(0);
        p.msg_id_mut()?
            .copy_from_slice(MessageId::INTERNAL_HEARTBEAT.as_bytes());
        p.payload_mut()?[0] = self.0;
        p.set_checksum(p.compute_checksum()?)?;
        Ok(())
    }

    pub fn decode_response<T: AsRef<[u8]>>(p: &Packet<T>) -> Result<Self, PacketProtocolError> {
        // TODO - sanity check protocol
        let hb = p.payload()?;
        Ok(hb[0].into())
    }
}
