use crate::client::AsyncClient;
use crate::EventChannel;
use crate::{Error, Frame, FrameKind, OpConfirm, QoS};

use async_trait::async_trait;
use std::collections::BTreeMap;
use std::sync::atomic;
use std::sync::Arc;
use std::time::Duration;
use tokio::sync::oneshot;
use tokio::sync::Mutex;
use tokio::task::JoinHandle;

use log::{error, warn};

const RPC_NOTIFICATION: u8 = 0x00;
const RPC_REQUEST: u8 = 0x01;
const RPC_REPLY: u8 = 0x11;
const RPC_ERROR: u8 = 0x12;

#[derive(Debug, Eq, PartialEq, Copy, Clone)]
#[repr(u8)]
pub enum EventKind {
    Notification = RPC_NOTIFICATION,
    Request = RPC_REQUEST,
    Reply = RPC_REPLY,
    ErrorReply = RPC_ERROR,
}

#[derive(Debug)]
pub struct Event {
    kind: EventKind,
    frame: Frame,
    payload_pos: usize,
    use_header: bool,
}

impl Event {
    #[inline]
    pub fn kind(&self) -> EventKind {
        self.kind
    }
    #[inline]
    pub fn frame(&self) -> &Frame {
        &self.frame
    }
    #[inline]
    pub fn sender(&self) -> &str {
        &self.frame.sender()
    }
    #[inline]
    pub fn payload(&self) -> &[u8] {
        &self.frame().payload()[self.payload_pos..]
    }
    /// # Panics
    ///
    /// Should not panic
    #[inline]
    pub fn id(&self) -> u32 {
        u32::from_le_bytes(
            if self.use_header {
                &self.frame.header().unwrap()[1..5]
            } else {
                &self.frame.payload()[1..5]
            }
            .try_into()
            .unwrap(),
        )
    }
    #[inline]
    pub fn method(&self) -> &[u8] {
        if self.use_header {
            &self.frame().header().unwrap()[5..]
        } else {
            &self.frame().payload()[5..self.payload_pos - 1]
        }
    }
    #[inline]
    pub fn parse_method(&self) -> Result<&str, Error> {
        std::str::from_utf8(self.method()).map_err(Into::into)
    }
    /// # Panics
    ///
    /// Should not panic
    #[inline]
    pub fn code(&self) -> i16 {
        if self.kind == EventKind::ErrorReply {
            i16::from_le_bytes(
                if self.use_header {
                    &self.frame.header().unwrap()[5..7]
                } else {
                    &self.frame.payload()[5..7]
                }
                .try_into()
                .unwrap(),
            )
        } else {
            0
        }
    }
}

impl TryFrom<Frame> for Event {
    type Error = Error;
    fn try_from(frame: Frame) -> Result<Self, Self::Error> {
        let (body, use_header) = frame
            .header()
            .map_or_else(|| (frame.payload(), false), |h| (h, true));
        if body.is_empty() {
            Err(Error::data("Empty RPC frame"))
        } else {
            macro_rules! check_len {
                ($len: expr) => {
                    if body.len() < $len {
                        return Err(Error::data("Invalid RPC frame"));
                    }
                };
            }
            match body[0] {
                RPC_NOTIFICATION => Ok(Event {
                    kind: EventKind::Notification,
                    frame,
                    payload_pos: if use_header { 0 } else { 1 },
                    use_header: false,
                }),
                RPC_REQUEST => {
                    check_len!(6);
                    if use_header {
                        Ok(Event {
                            kind: EventKind::Request,
                            frame,
                            payload_pos: 0,
                            use_header: true,
                        })
                    } else {
                        let mut sp = body[5..].splitn(2, |c| *c == 0);
                        let method = sp.next().ok_or_else(|| Error::data("No RPC method"))?;
                        let payload_pos = 6 + method.len();
                        sp.next()
                            .ok_or_else(|| Error::data("No RPC params block"))?;
                        Ok(Event {
                            kind: EventKind::Request,
                            frame,
                            payload_pos,
                            use_header: false,
                        })
                    }
                }
                RPC_REPLY => {
                    check_len!(5);
                    Ok(Event {
                        kind: EventKind::Reply,
                        frame,
                        payload_pos: if use_header { 0 } else { 5 },
                        use_header,
                    })
                }
                RPC_ERROR => {
                    check_len!(7);
                    Ok(Event {
                        kind: EventKind::ErrorReply,
                        frame,
                        payload_pos: if use_header { 0 } else { 7 },
                        use_header,
                    })
                }
                v => Err(Error::data(format!("Unsupported RPC frame code {}", v))),
            }
        }
    }
}

#[async_trait]
pub trait RpcHandlers {
    async fn handle_call(&self, event: Event) -> RpcResult;
    async fn handle_notification(&self, event: Event);
    async fn handle_frame(&self, frame: Frame);
}

pub struct DummyHandlers {}

#[async_trait]
impl RpcHandlers for DummyHandlers {
    async fn handle_call(&self, _event: Event) -> RpcResult {
        Err(RpcError::new(
            -32000,
            Some("RPC handler is not set".as_bytes().to_vec()),
        ))
    }
    async fn handle_notification(&self, _event: Event) {}
    async fn handle_frame(&self, _frame: Frame) {}
}

type CallMap = Arc<std::sync::Mutex<BTreeMap<u32, oneshot::Sender<Event>>>>;

pub struct Rpc {
    call_id: u32,
    timeout: Option<Duration>,
    client: Arc<Mutex<dyn AsyncClient>>,
    processor_fut: Arc<std::sync::Mutex<JoinHandle<()>>>,
    pinger_fut: Option<JoinHandle<()>>,
    calls: CallMap,
    connected: Option<Arc<atomic::AtomicBool>>,
}

async fn processor<C, H>(
    rx: EventChannel,
    processor_client: Arc<Mutex<C>>,
    calls: CallMap,
    handlers: &'static H,
) where
    C: AsyncClient + 'static,
    H: RpcHandlers + Send + Sync,
{
    while let Ok(frame) = rx.recv().await {
        if frame.kind() == FrameKind::Message {
            match TryInto::<Event>::try_into(frame) {
                Ok(event) => match event.kind() {
                    EventKind::Notification => {
                        tokio::spawn(handlers.handle_notification(event));
                    }
                    EventKind::Request => {
                        let id = event.id();
                        let ev = if id > 0 {
                            Some((event.frame().sender().to_owned(), processor_client.clone()))
                        } else {
                            None
                        };
                        tokio::spawn(async move {
                            let res = handlers.handle_call(event).await;
                            if let Some(ev) = ev {
                                macro_rules! send_reply {
                                    ($payload: expr, $result: expr) => {{
                                        let mut client = ev.1.lock().await;
                                        if let Some(result) = $result {
                                            client
                                                .zc_send(
                                                    &ev.0,
                                                    $payload,
                                                    result.into(),
                                                    QoS::Processed,
                                                )
                                                .await
                                        } else {
                                            client
                                                .zc_send(
                                                    &ev.0,
                                                    $payload,
                                                    (&[][..]).into(),
                                                    QoS::Processed,
                                                )
                                                .await
                                        }
                                    }};
                                }
                                match res {
                                    Ok(v) => {
                                        let mut payload = Vec::with_capacity(5);
                                        payload.push(RPC_REPLY);
                                        payload.extend_from_slice(&id.to_le_bytes());
                                        let _r = send_reply!(payload.into(), v);
                                    }
                                    Err(e) => {
                                        let mut payload = Vec::with_capacity(7);
                                        payload.push(RPC_ERROR);
                                        payload.extend_from_slice(&id.to_le_bytes());
                                        payload.extend_from_slice(&e.code.to_le_bytes());
                                        let _r = send_reply!(payload.into(), e.data);
                                    }
                                }
                            }
                        });
                    }
                    EventKind::Reply | EventKind::ErrorReply => {
                        let id = event.id();
                        if let Some(tx) = { calls.lock().unwrap().remove(&id) } {
                            let _r = tx.send(event);
                        } else {
                            warn!("orphaned rpc response: {}", id);
                        }
                    }
                },
                Err(e) => {
                    error!("{}", e);
                }
            }
        } else {
            handlers.handle_frame(frame).await;
        }
    }
}

#[inline]
fn prepare_call_payload(method: &str, id_bytes: &[u8]) -> Vec<u8> {
    let m = method.as_bytes();
    let mut payload = Vec::with_capacity(m.len() + 6);
    payload.push(RPC_REQUEST);
    payload.extend(id_bytes);
    payload.extend(m);
    payload.push(0x00);
    payload
}

impl Rpc {
    /// # Panics
    ///
    /// Should not panic
    pub async fn new<H>(
        mut client: impl AsyncClient + 'static,
        handlers: &'static H,
    ) -> Result<Self, Error>
    where
        H: RpcHandlers + Send + Sync,
    {
        let timeout = client.get_timeout();
        let rx = { client.take_event_channel().unwrap() };
        let connected = client.get_connected_beacon();
        let client = Arc::new(Mutex::new(client));
        let calls: CallMap = <_>::default();
        let processor_fut = Arc::new(std::sync::Mutex::new(tokio::spawn(processor(
            rx,
            client.clone(),
            calls.clone(),
            handlers,
        ))));
        let pinger_client = client.clone();
        let pfut = processor_fut.clone();
        let pinger_fut = timeout.map(|t| {
            tokio::spawn(async move {
                loop {
                    if let Err(e) = pinger_client.lock().await.ping().await {
                        error!("{}", e);
                        pfut.lock().unwrap().abort();
                        break;
                    }
                    tokio::time::sleep(t).await;
                }
            })
        });
        Ok(Self {
            call_id: 0,
            timeout,
            client,
            processor_fut,
            pinger_fut,
            calls,
            connected,
        })
    }
    #[inline]
    pub fn client(&self) -> Arc<Mutex<(dyn AsyncClient + 'static)>> {
        self.client.clone()
    }
    #[inline]
    pub async fn notify(
        &mut self,
        target: &str,
        data: &[u8],
        qos: QoS,
    ) -> Result<OpConfirm, Error> {
        self.client
            .lock()
            .await
            .zc_send(target, (&[RPC_NOTIFICATION][..]).into(), data.into(), qos)
            .await
    }
    pub async fn call0(
        &self,
        target: &str,
        method: &str,
        params: &[u8],
    ) -> Result<OpConfirm, Error> {
        let payload = prepare_call_payload(method, &[0, 0, 0, 0]);
        self.client
            .lock()
            .await
            .zc_send(target, payload.into(), params.into(), QoS::Processed)
            .await
    }
    /// # Panics
    ///
    /// Will panic on poisoned mutex
    pub async fn call(
        &mut self,
        target: &str,
        method: &str,
        params: &[u8],
    ) -> Result<Event, Error> {
        self.increase_call_id();
        let call_id = self.call_id;
        let payload = prepare_call_payload(method, &call_id.to_le_bytes());
        let (tx, rx) = oneshot::channel();
        self.calls.lock().unwrap().insert(call_id, tx);
        macro_rules! unwrap_or_cancel {
            ($result: expr) => {
                match $result {
                    Ok(v) => v,
                    Err(e) => {
                        self.calls.lock().unwrap().remove(&call_id);
                        return Err(Into::into(e));
                    }
                }
            };
        }
        unwrap_or_cancel!(unwrap_or_cancel!(
            unwrap_or_cancel!(
                self.client
                    .lock()
                    .await
                    .zc_send(target, payload.into(), params.into(), QoS::Processed)
                    .await
            )
            .unwrap()
            .await
        ));
        if let Some(timeout) = self.timeout {
            Ok(tokio::time::timeout(timeout, rx).await??)
        } else {
            Ok(rx.await?)
        }
    }
    #[inline]
    fn increase_call_id(&mut self) {
        if self.call_id == u32::MAX {
            self.call_id = 1;
        } else {
            self.call_id += 1;
        }
    }
    pub async fn is_connected(&self) -> bool {
        self.connected
            .as_ref()
            .map_or(true, |b| b.load(atomic::Ordering::SeqCst))
    }
}

impl Drop for Rpc {
    fn drop(&mut self) {
        self.pinger_fut.as_ref().map(JoinHandle::abort);
        self.processor_fut.lock().unwrap().abort();
    }
}

#[derive(Debug)]
pub struct RpcError {
    code: i16,
    data: Option<Vec<u8>>,
}

impl RpcError {
    pub fn new(code: i16, data: Option<Vec<u8>>) -> Self {
        Self { code, data }
    }
}

pub type RpcResult = Result<Option<Vec<u8>>, RpcError>;
