use log::{error, info, trace};
use std::collections::{hash_map, HashMap};
use std::fmt;
use std::hash::{Hash, Hasher};
use std::marker::Unpin;
use std::sync::atomic;
use std::sync::Arc;
use std::sync::RwLock;
use std::time::Duration;
use submap::{BroadcastMap, SubMap};
use tokio::io::{AsyncReadExt, AsyncWriteExt, BufReader, BufWriter};
use tokio::net::{TcpListener, TcpStream, UnixListener, UnixStream};
use tokio::task::JoinHandle;
use tokio::time;

use crate::{Error, ErrorKind, GREETINGS, PROTOCOL_VERSION};

use crate::ERR_NOT_SUPPORTED;
use crate::RESPONSE_OK;

use crate::OP_ACK;

use crate::borrow::Cow;
use crate::client::AsyncClient;
use crate::{EventChannel, OpConfirm};
use crate::{Frame, FrameData, FrameKind, FrameOp, QoS};

use async_trait::async_trait;

const DEFAULT_QUEUE_SIZE: usize = 8192;

macro_rules! pretty_error {
    ($name: expr, $err:expr) => {
        if $err.kind() != ErrorKind::Eof {
            error!("client {} error: {}", $name, $err);
        }
    };
}

type BrokerClient = Arc<ElbusClient>;

macro_rules! make_confirm_channel {
    ($qos: expr) => {
        match $qos {
            QoS::No => Ok(None),
            QoS::Processed => {
                let (tx, rx) = tokio::sync::oneshot::channel();
                let _r = tx.send(Ok(()));
                Ok(Some(rx))
            }
        }
    };
}

macro_rules! send {
    ($db:expr, $client:expr, $target:expr, $header: expr, $buf:expr, $payload_pos:expr) => {{
        trace!("elbus message from {} to {}", $client, $target);
        let tx = {
            $db.clients
                .read()
                .unwrap()
                .get($target)
                .map(|c| c.tx.clone())
        };
        if let Some(tx) = tx {
            let frame = Arc::new(FrameData {
                kind: FrameKind::Message,
                sender: Some($client.name.clone()),
                topic: None,
                header: $header,
                buf: $buf,
                payload_pos: $payload_pos,
            });
            tx.send(frame).await.map_err(Into::into)
        } else {
            Err(Error::not_registered())
        }
    }};
}

macro_rules! send_broadcast {
    ($db:expr, $client:expr, $target:expr, $header: expr, $buf:expr, $payload_pos:expr) => {{
        trace!("elbus broadcast message from {} to {}", $client, $target);
        let subs = { $db.broadcasts.read().unwrap().get_clients_by_mask($target) };
        if !subs.is_empty() {
            let frame = Arc::new(FrameData {
                kind: FrameKind::Broadcast,
                sender: Some($client.name.clone()),
                topic: None,
                header: $header,
                buf: $buf,
                payload_pos: $payload_pos,
            });
            for sub in subs {
                let _r = sub.tx.send(frame.clone()).await;
            }
        }
    }};
}

macro_rules! publish {
    ($db:expr, $client:expr, $topic:expr, $header: expr, $buf:expr, $payload_pos:expr) => {{
        trace!("elbus topic publish from {} to {}", $client, $topic);
        let subs = { $db.subscriptions.read().unwrap().get_subscribers($topic) };
        if !subs.is_empty() {
            let frame = Arc::new(FrameData {
                kind: FrameKind::Publish,
                sender: Some($client.name.clone()),
                topic: Some($topic.to_owned()),
                header: $header,
                buf: $buf,
                payload_pos: $payload_pos,
            });
            for sub in subs {
                let _r = sub.tx.send(frame.clone()).await;
            }
        }
    }};
}

pub struct Client {
    client: Arc<ElbusClient>,
    db: Arc<BrokerDb>,
    rx: Option<EventChannel>,
}

#[async_trait]
impl AsyncClient for Client {
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    async fn subscribe(&mut self, topic: &str, qos: QoS) -> Result<OpConfirm, Error> {
        if self
            .db
            .subscriptions
            .write()
            .unwrap()
            .subscribe(topic, &self.client)
        {
            make_confirm_channel!(qos)
        } else {
            Err(Error::not_registered())
        }
    }
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    async fn subscribe_bulk(&mut self, topics: Vec<&str>, qos: QoS) -> Result<OpConfirm, Error> {
        let mut db = self.db.subscriptions.write().unwrap();
        for topic in topics {
            if !db.subscribe(topic, &self.client) {
                return Err(Error::not_registered());
            }
        }
        make_confirm_channel!(qos)
    }
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    async fn unsubscribe(&mut self, topic: &str, qos: QoS) -> Result<OpConfirm, Error> {
        if self
            .db
            .subscriptions
            .write()
            .unwrap()
            .unsubscribe(topic, &self.client)
        {
            make_confirm_channel!(qos)
        } else {
            Err(Error::not_registered())
        }
    }
    /// # Panics
    ///
    /// Will panic if the mutex is poisoned
    async fn unsubscribe_bulk(&mut self, topics: Vec<&str>, qos: QoS) -> Result<OpConfirm, Error> {
        let mut db = self.db.subscriptions.write().unwrap();
        for topic in topics {
            if !db.unsubscribe(topic, &self.client) {
                return Err(Error::not_registered());
            }
        }
        make_confirm_channel!(qos)
    }
    #[inline]
    async fn send(
        &mut self,
        target: &str,
        payload: Cow<'async_trait>,
        qos: QoS,
    ) -> Result<OpConfirm, Error> {
        send!(self.db, self.client, target, None, payload.to_vec(), 0)?;
        make_confirm_channel!(qos)
    }
    #[inline]
    async fn zc_send(
        &mut self,
        target: &str,
        header: Cow<'async_trait>,
        payload: Cow<'async_trait>,
        qos: QoS,
    ) -> Result<OpConfirm, Error> {
        send!(
            self.db,
            self.client,
            target,
            Some(header.to_vec()),
            payload.to_vec(),
            0
        )?;
        make_confirm_channel!(qos)
    }
    #[inline]
    async fn send_broadcast(
        &mut self,
        target: &str,
        payload: Cow<'async_trait>,
        qos: QoS,
    ) -> Result<OpConfirm, Error> {
        send_broadcast!(self.db, self.client, target, None, payload.to_vec(), 0);
        make_confirm_channel!(qos)
    }
    #[inline]
    async fn publish(
        &mut self,
        topic: &str,
        payload: Cow<'async_trait>,
        qos: QoS,
    ) -> Result<OpConfirm, Error> {
        publish!(self.db, self.client, topic, None, payload.to_vec(), 0);
        make_confirm_channel!(qos)
    }
    #[inline]
    fn take_event_channel(&mut self) -> Option<EventChannel> {
        self.rx.take()
    }
    #[inline]
    async fn ping(&mut self) -> Result<(), Error> {
        Ok(())
    }
    #[inline]
    fn is_connected(&self) -> bool {
        true
    }
    #[inline]
    fn get_timeout(&self) -> Option<Duration> {
        None
    }
    #[inline]
    fn get_connected_beacon(&self) -> Option<Arc<atomic::AtomicBool>> {
        None
    }
}

impl Client {
    #[inline]
    fn unregister(&self) {
        self.db.unregister_client(&self.client);
    }
}

impl Drop for Client {
    fn drop(&mut self) {
        self.unregister();
    }
}

#[derive(Debug)]
struct ElbusClient {
    name: String,
    tx: async_channel::Sender<Frame>,
}

impl fmt::Display for ElbusClient {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.name)
    }
}

impl ElbusClient {
    pub fn new(name: &str, queue_size: usize) -> (Self, EventChannel) {
        let (tx, rx) = async_channel::bounded(queue_size);
        (
            Self {
                name: name.to_owned(),
                tx,
            },
            rx,
        )
    }
}

impl PartialEq for ElbusClient {
    fn eq(&self, other: &Self) -> bool {
        self.name == other.name
    }
}

impl Eq for ElbusClient {}

impl Hash for ElbusClient {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.name.hash(state);
    }
}

struct BrokerDb {
    clients: RwLock<HashMap<String, BrokerClient>>,
    broadcasts: RwLock<BroadcastMap<BrokerClient>>,
    subscriptions: RwLock<SubMap<BrokerClient>>,
}

impl Default for BrokerDb {
    fn default() -> Self {
        Self {
            clients: <_>::default(),
            broadcasts: RwLock::new(
                BroadcastMap::new()
                    .separator('.')
                    .match_any("?")
                    .wildcard("*"),
            ),
            subscriptions: RwLock::new(SubMap::new().separator('/').match_any("+").wildcard("#")),
        }
    }
}

impl BrokerDb {
    fn register_client(&self, client: Arc<ElbusClient>) -> Result<(), Error> {
        if let hash_map::Entry::Vacant(x) = self.clients.write().unwrap().entry(client.name.clone())
        {
            {
                let mut bdb = self.broadcasts.write().unwrap();
                bdb.register_client(&client.name, &client);
            }
            {
                let mut sdb = self.subscriptions.write().unwrap();
                sdb.register_client(&client);
                sdb.subscribe("!bus", &client);
                sdb.subscribe("!bus/#", &client);
            }
            x.insert(client);
            Ok(())
        } else {
            Err(Error::busy("client already registred"))
        }
    }
    fn unregister_client(&self, client: &Arc<ElbusClient>) {
        self.subscriptions
            .write()
            .unwrap()
            .unregister_client(client);
        self.broadcasts
            .write()
            .unwrap()
            .unregister_client(&client.name, client);
        self.clients.write().unwrap().remove(&client.name);
    }
}

pub struct Broker {
    db: Arc<BrokerDb>,
    services: Vec<JoinHandle<()>>,
    queue_size: usize,
}

impl Default for Broker {
    fn default() -> Self {
        Self {
            db: <_>::default(),
            services: <_>::default(),
            queue_size: DEFAULT_QUEUE_SIZE,
        }
    }
}

#[allow(clippy::unnecessary_wraps)]
#[inline]
fn prepare_unix_stream(_stream: &UnixStream) -> Result<(), Error> {
    Ok(())
}

#[inline]
fn prepare_tcp_stream(stream: &TcpStream) -> Result<(), Error> {
    stream.set_nodelay(true).map_err(Into::into)
}

macro_rules! spawn_server {
    ($self: expr, $path: expr, $listener: expr, $buf_size: expr, $timeout: expr,
     $prepare: ident) => {{
        let socket_path = $path.to_owned();
        let db = $self.db.clone();
        let queue_size = $self.queue_size;
        let service = tokio::spawn(async move {
            loop {
                match $listener.accept().await {
                    Ok((stream, addr)) => {
                        trace!(
                            "elbus tcp client connected from {:?} to {}",
                            addr,
                            socket_path
                        );
                        if let Err(e) = $prepare(&stream) {
                            error!("{}", e);
                            continue;
                        }
                        let (reader, writer) = stream.into_split();
                        let reader = BufReader::with_capacity($buf_size, reader);
                        let writer = BufWriter::with_capacity($buf_size, writer);
                        let cdb = db.clone();
                        let name = socket_path.clone();
                        tokio::spawn(async move {
                            if let Err(e) =
                                Self::handle_peer(cdb, reader, writer, $timeout, queue_size).await
                            {
                                pretty_error!(name, e);
                            }
                        });
                    }
                    Err(e) => error!("{}", e),
                }
            }
        });
        $self.services.push(service);
    }};
}

impl Broker {
    pub fn new() -> Self {
        Self::default()
    }
    pub async fn register_client(&self, name: &str) -> Result<Client, Error> {
        let (c, rx) = ElbusClient::new(name, self.queue_size);
        let client = Arc::new(c);
        self.db.register_client(client.clone())?;
        Ok(Client {
            client,
            db: self.db.clone(),
            rx: Some(rx),
        })
    }
    pub async fn spawn_unix_server(
        &mut self,
        path: &str,
        buf_size: usize,
        timeout: Duration,
    ) -> Result<(), Error> {
        let _r = tokio::fs::remove_file(path).await;
        let listener = UnixListener::bind(path)?;
        spawn_server!(self, path, listener, buf_size, timeout, prepare_unix_stream);
        Ok(())
    }
    pub async fn spawn_tcp_server(
        &mut self,
        path: &str,
        buf_size: usize,
        timeout: Duration,
    ) -> Result<(), Error> {
        let listener = TcpListener::bind(path).await?;
        spawn_server!(self, path, listener, buf_size, timeout, prepare_tcp_stream);
        Ok(())
    }
    #[allow(clippy::items_after_statements)]
    pub async fn spawn_fifo(&mut self, path: &str) -> Result<(), Error> {
        let _r = tokio::fs::remove_file(path).await;
        unix_named_pipe::create(path, Some(0o622))?;
        use std::os::unix::fs::PermissionsExt;
        use tokio::io::AsyncBufReadExt;
        // chown fifo as it's usually created with 644
        tokio::fs::set_permissions(path, std::fs::Permissions::from_mode(0o622)).await?;
        let fd = unix_named_pipe::open_read(path)?;
        //let socket_path = path.to_owned();
        let service = tokio::spawn(async move {
            let f = tokio::fs::File::from_std(fd);
            let reader = BufReader::new(f);
            let mut _lines = reader.lines();
            //loop {
            // TODO put rpc call
            //while let Some(line) = lines.next_line().await.unwrap() {
            //println!("{}", line);
            //}
            //}
        });
        self.services.push(service);
        Ok(())
    }
    #[allow(clippy::too_many_lines)]
    async fn handle_peer<R, W>(
        db: Arc<BrokerDb>,
        mut reader: R,
        mut writer: W,
        timeout: Duration,
        queue_size: usize,
    ) -> Result<(), Error>
    where
        R: AsyncReadExt + Unpin,
        W: AsyncWriteExt + Unpin + Send + 'static,
    {
        macro_rules! write_and_flush {
            ($buf: expr) => {
                time::timeout(timeout, writer.write_all($buf)).await??;
                time::timeout(timeout, writer.flush()).await??;
            };
        }
        let mut buf = GREETINGS.to_vec();
        buf.extend_from_slice(&PROTOCOL_VERSION.to_le_bytes());
        write_and_flush!(&buf);
        let mut buf = vec![0; 3];
        time::timeout(timeout, reader.read_exact(&mut buf)).await??;
        if buf[0] != GREETINGS[0] {
            write_and_flush!(&[ERR_NOT_SUPPORTED]);
            return Err(Error::not_supported("invalid protocol"));
        }
        if u16::from_le_bytes(buf[1..3].try_into().unwrap()) != PROTOCOL_VERSION {
            write_and_flush!(&[ERR_NOT_SUPPORTED]);
            return Err(Error::not_supported("unsupported protocol version"));
        }
        write_and_flush!(&[RESPONSE_OK]);
        let mut buf = vec![0; 2];
        time::timeout(timeout, reader.read_exact(&mut buf)).await??;
        let len = u16::from_le_bytes(buf.try_into().unwrap());
        let mut buf = vec![0; len as usize];
        time::timeout(timeout, reader.read_exact(&mut buf)).await??;
        let client_name = std::str::from_utf8(&buf)?.to_owned();
        if client_name.is_empty() || client_name == "!bus" {
            return Err(Error::data("Invalid client name"));
        }
        let (client, rx) = {
            let (c, rx) = ElbusClient::new(&client_name, queue_size);
            let client = Arc::new(c);
            if let Err(e) = db.register_client(client.clone()) {
                write_and_flush!(&[e.kind as u8]);
                return Err(e);
            }
            write_and_flush!(&[RESPONSE_OK]);
            (client, rx)
        };
        info!("elbus client registered: {}", client_name);
        let w_name = client_name.clone();
        let writer_fut = tokio::spawn(async move {
            while let Ok(frame) = rx.recv().await {
                macro_rules! write_data {
                    ($data: expr) => {
                        if !$data.is_empty() {
                            match time::timeout(timeout, writer.write_all($data)).await {
                                Ok(result) => {
                                    if let Err(e) = result {
                                        pretty_error!(w_name, Into::<Error>::into(&e));
                                        break;
                                    }
                                }
                                Err(_) => {
                                    error!("client {} error: timeout", w_name);
                                    break;
                                }
                            }
                        }
                    };
                }
                macro_rules! flush {
                    () => {
                        match time::timeout(timeout, writer.flush()).await {
                            Ok(result) => {
                                if let Err(e) = result {
                                    pretty_error!(w_name, Into::<Error>::into(&e));
                                    break;
                                }
                            }
                            Err(_) => {
                                error!("client {} error: timeout", w_name);
                                break;
                            }
                        }
                    };
                }
                if frame.kind == FrameKind::Prepared {
                    write_data!(&frame.buf);
                    flush!();
                } else {
                    let sender = frame.sender.as_ref().unwrap().as_bytes();
                    let topic = frame.topic.as_ref().map(String::as_bytes);
                    let mut extra_len = sender.len();
                    if let Some(t) = topic.as_ref() {
                        extra_len += t.len() + 1;
                    }
                    let mut buf = Vec::with_capacity(7 + extra_len);
                    buf.push(frame.kind as u8); // byte 0
                    let frame_len = extra_len + frame.buf.len() - frame.payload_pos + 1;
                    #[allow(clippy::cast_possible_truncation)]
                    buf.extend_from_slice(&(frame_len as u32).to_le_bytes()); // bytes 1-4
                    buf.push(0x00); // byte 5 - reserved
                    buf.extend_from_slice(sender);
                    buf.push(0x00);
                    if let Some(t) = topic.as_ref() {
                        buf.extend_from_slice(t);
                        buf.push(0x00);
                    };
                    write_data!(&buf);
                    if let Some(header) = frame.header() {
                        write_data!(header);
                    }
                    write_data!(frame.payload());
                    flush!();
                }
            }
        });
        let result = Self::handle_reader(&db, client.clone(), &mut reader, timeout).await;
        writer_fut.abort();
        db.unregister_client(&client);
        info!("elbus client disconnected: {}", client_name);
        result
    }

    // TODO send ack only after the client received message (QoS2)
    #[allow(clippy::too_many_lines)]
    async fn handle_reader<R>(
        db: &BrokerDb,
        client: Arc<ElbusClient>,
        reader: &mut R,
        timeout: Duration,
    ) -> Result<(), Error>
    where
        R: AsyncReadExt + Unpin,
    {
        loop {
            let mut buf = vec![0; 9];
            reader.read_exact(&mut buf).await?;
            let flags = buf[4];
            if flags == 0 {
                // OP_NOP
                trace!("{} ping", client);
                continue;
            }
            let op_id = &buf[0..4];
            let op: FrameOp = (flags & 0b0011_1111).try_into()?;
            let qos: QoS = (flags >> 6 & 0b0011_1111).try_into()?;
            let len = u32::from_le_bytes(buf[5..9].try_into().unwrap());
            let mut buf = vec![0; len as usize];
            time::timeout(timeout, reader.read_exact(&mut buf)).await??;
            macro_rules! send_ack {
                ($code:expr) => {
                    let mut buf = Vec::with_capacity(6);
                    buf.push(OP_ACK);
                    buf.extend_from_slice(op_id);
                    buf.push($code);
                    client
                        .tx
                        .send(Arc::new(FrameData {
                            kind: FrameKind::Prepared,
                            sender: None,
                            topic: None,
                            header: None,
                            buf,
                            payload_pos: 0,
                        }))
                        .await?;
                };
            }
            match op {
                FrameOp::SubscribeTopic => {
                    let sp = buf.split(|c| *c == 0);
                    {
                        let mut sdb = db.subscriptions.write().unwrap();
                        for t in sp {
                            let topic = std::str::from_utf8(t)?;
                            sdb.subscribe(topic, &client);
                            trace!("elbus client {} subscribed to topic {}", client, topic);
                        }
                    }
                    if qos == QoS::Processed {
                        send_ack!(RESPONSE_OK);
                    }
                }
                FrameOp::UnsubscribeTopic => {
                    let sp = buf.split(|c| *c == 0);
                    {
                        let mut sdb = db.subscriptions.write().unwrap();
                        for t in sp {
                            let topic = std::str::from_utf8(t)?;
                            sdb.unsubscribe(topic, &client);
                            trace!("elbus client {} unsubscribed from topic {}", client, topic);
                        }
                    }
                    if qos == QoS::Processed {
                        send_ack!(RESPONSE_OK);
                    }
                }
                _ => {
                    let mut sp = buf.splitn(2, |c| *c == 0);
                    let tgt = sp.next().ok_or_else(|| Error::data("broken frame"))?;
                    let target = std::str::from_utf8(tgt)?;
                    sp.next().ok_or_else(|| Error::data("broken frame"))?;
                    let payload_pos = tgt.len() + 1;
                    drop(sp);
                    match op {
                        FrameOp::Message => {
                            if let Err(e) = send!(db, client, target, None, buf, payload_pos) {
                                send_ack!(e.kind as u8);
                            } else if qos == QoS::Processed {
                                send_ack!(RESPONSE_OK);
                            }
                        }
                        FrameOp::Broadcast => {
                            send_broadcast!(db, client, target, None, buf, payload_pos);
                            if qos == QoS::Processed {
                                send_ack!(RESPONSE_OK);
                            }
                        }
                        FrameOp::PublishTopic => {
                            publish!(db, client, target, None, buf, payload_pos);
                            if qos == QoS::Processed {
                                send_ack!(RESPONSE_OK);
                            }
                        }
                        _ => {}
                    }
                }
            }
        }
    }
}

impl Drop for Broker {
    fn drop(&mut self) {
        for service in &self.services {
            service.abort();
        }
    }
}
