use std::sync::Arc;

pub enum Cow<'a> {
    Borrowed(&'a [u8]),
    Owned(Vec<u8>),
    Referenced(Arc<Vec<u8>>),
}

impl<'a> From<Vec<u8>> for Cow<'a> {
    fn from(src: Vec<u8>) -> Cow<'a> {
        Cow::Owned(src)
    }
}

impl<'a> From<Arc<Vec<u8>>> for Cow<'a> {
    fn from(src: Arc<Vec<u8>>) -> Cow<'a> {
        Cow::Referenced(src)
    }
}

impl<'a> From<&'a [u8]> for Cow<'a> {
    fn from(src: &'a [u8]) -> Cow<'a> {
        Cow::Borrowed(src)
    }
}

impl<'a> Cow<'a> {
    pub fn as_slice(&self) -> &[u8] {
        match self {
            Cow::Borrowed(v) => v,
            Cow::Owned(v) => v.as_slice(),
            Cow::Referenced(v) => v.as_slice(),
        }
    }
    pub fn to_vec(self) -> Vec<u8> {
        match self {
            Cow::Borrowed(v) => v.to_vec(),
            Cow::Owned(v) => v,
            Cow::Referenced(v) => v.to_vec(),
        }
    }
    pub fn len(&self) -> usize {
        match self {
            Cow::Borrowed(v) => v.len(),
            Cow::Owned(v) => v.len(),
            Cow::Referenced(v) => v.len(),
        }
    }
    pub fn is_empty(&self) -> bool {
        match self {
            Cow::Borrowed(v) => v.is_empty(),
            Cow::Owned(v) => v.is_empty(),
            Cow::Referenced(v) => v.is_empty(),
        }
    }
}
