use async_trait::async_trait;
use hyper::client::HttpConnector;
use hyper::Client;
use hyper_tls::HttpsConnector;

mod error;
pub use error::Error;
pub use http;

pub mod content;
pub use content::{RequestContent, ResponseContent};

#[cfg(feature = "sse")]
pub mod sse;

#[cfg(feature = "sse")]
pub use sse::Sse;

#[async_trait]
pub trait HttpClient
where
    Self: Send + Sync + Sized,
{
    async fn perform(
        &self,
        r: hyper::Request<hyper::Body>,
    ) -> Result<hyper::Response<hyper::Body>, hyper::Error>;

    async fn send<Req, Res>(
        &self,
        r: http::Request<Req>,
    ) -> Result<http::Response<Res::Data>, Error>
    where
        Req: RequestContent + Send,
        Res: ResponseContent,
    {
        let (mut parts, body) = r.into_parts();

        body.apply_headers(&mut parts.headers);

        let hyper_body: hyper::Body = body.into_body()?;
        let request = hyper::Request::from_parts(parts, hyper_body);

        let res = self.perform(request).await?;

        Res::convert_response(res).await
    }

    fn https() -> HttpsClient {
        HttpsClient::default()
    }
}

pub struct HttpsClient {
    http_client: Client<HttpsConnector<HttpConnector>>,
}

impl Default for HttpsClient {
    fn default() -> Self {
        let http_client = Client::builder().build::<_, hyper::Body>(HttpsConnector::new());
        Self { http_client }
    }
}

#[async_trait]
impl HttpClient for HttpsClient {
    async fn perform(
        &self,
        r: hyper::Request<hyper::Body>,
    ) -> Result<hyper::Response<hyper::Body>, hyper::Error> {
        self.http_client.request(r).await
    }
}
