use crate::Renderer;

pub struct RenderTarget {
    texture: wgpu::Texture,
    view: wgpu::TextureView,

    texture_id: egui::TextureId,
}

impl RenderTarget {
    pub fn new(
        device: &wgpu::Device,
        renderer: &mut Renderer,
        texture_format: wgpu::TextureFormat,
        size: [u32; 2],
    ) -> Self {
        let extent = wgpu::Extent3d {
            width: size[0],
            height: size[1],
            depth_or_array_layers: 1,
        };

        let descriptor = wgpu::TextureDescriptor {
            label: Some("egui_render_target_texture"),
            size: extent,
            mip_level_count: 1,
            sample_count: 1,
            dimension: wgpu::TextureDimension::D2,
            format: texture_format,
            usage: wgpu::TextureUsage::SAMPLED | wgpu::TextureUsage::RENDER_ATTACHMENT,
        };

        let texture = device.create_texture(&descriptor);
        let view = texture.create_view(&wgpu::TextureViewDescriptor::default());

        let gui_wgpu_texture_layout_descriptor = wgpu::BindGroupLayoutDescriptor {
            label: Some("gui-wgpu bind group layout"),
            entries: &[
                wgpu::BindGroupLayoutEntry {
                    binding: 0,
                    visibility: wgpu::ShaderStage::FRAGMENT,
                    ty: wgpu::BindingType::Texture {
                        multisampled: false,
                        sample_type: wgpu::TextureSampleType::Float { filterable: true },
                        view_dimension: wgpu::TextureViewDimension::D2,
                    },
                    count: None,
                },
                wgpu::BindGroupLayoutEntry {
                    binding: 1,
                    visibility: wgpu::ShaderStage::FRAGMENT,
                    ty: wgpu::BindingType::Sampler {
                        comparison: false,
                        filtering: true,
                    },
                    count: None,
                },
            ],
        };

        let _bind_group_layout =
            device.create_bind_group_layout(&gui_wgpu_texture_layout_descriptor);

        let texture_id = renderer.egui_texture_from_wgpu_texture(device, &texture);

        Self {
            texture,
            view,

            texture_id,
        }
    }

    pub fn texture_id(&self) -> egui::TextureId {
        self.texture_id
    }
    pub fn texture(&self) -> &wgpu::Texture {
        &self.texture
    }
    pub fn view(&self) -> &wgpu::TextureView {
        &self.view
    }
}
