#![warn(rust_2018_idioms)]
#![allow(unused_imports)]

mod file_io;
mod prelude;
mod process;

use file_io::*;
use prelude::*;
use process::*;
use rayon::iter::{IntoParallelRefIterator, ParallelIterator};

use indicatif::*;
use std::error::Error;
use std::io::*;
use std::path::Path;

fn main() {
    if std::env::args().len() < 2 {
        println!("erreur: il faut passer en argument le(s) sujet(s) à traiter !");
        std::process::exit(1);
    }

    let heuristics = match read_heuristics(Path::new(""), Path::new("heuristiques.toml")) {
        Err(err) => {
            print_error(&err);
            std::process::exit(1)
        }
        Ok(res) => res,
    };

    println!("Annulations : {:?}.\n", heuristics.strategy);
    println!("Coeff. max. : {:?}.\n", heuristics.max_coefficient);

    for file in std::env::args().skip(1) {
        if let Err(err) = process(&file, &heuristics) {
            print_error(&err);
            std::process::exit(1);
        }
    }
}

fn process(file: &str, heuristics: &Heuristics) -> Result<Vec<Vec<Axis>>> {
    println!("Traitement du sujet {}.\n", file);

    let out = Path::new("sorties").join(Path::new(&format!("{}.tsv", file)));

    write(&Vec::new(), &out).expect("Erreur lors de l'écriture du fichier.");

    let data = read("data", file).unwrap_or_else(|_| panic!("Erreur d'entrée pour {}", file));
    let style = ProgressStyle::default_bar()
        .template("[{elapsed_precise}] {wide_bar} [{eta_precise}]")
        .progress_chars("░▓▒");

    let axes = build_axes(&data, heuristics);

    let mut partsols = vec![PartSol(Vec::new())];

    println!("profondeur  largeur\n-------------------");

    for (j, candidate) in data.candidates.iter().enumerate() {
        let delta = candidate.actual_error;
        let wunderbar = ProgressBar::new(partsols.len() as u64);
        wunderbar.set_style(style.clone());
        partsols = partsols
            .par_iter()
            .progress_with(wunderbar)
            .map(|s| extensions(s, &axes, delta))
            .flatten()
            .collect();
        println!("{:<9}{}", j + 1, partsols.len());
    }

    println!("-------------------\n");

    let sols: Vec<Vec<Axis>> = partsols.iter().map(|s| s.restrict(&axes)).collect();

    write(&sols, &out).expect("Erreur lors de l'écriture du fichier.");

    Ok(sols)
}

fn print_error(mut err: &dyn Error) {
    let _ = writeln!(stderr(), "error: {}", err);
    while let Some(cause) = err.source() {
        let _ = writeln!(stderr(), "caused by: {}", cause);
        err = cause;
    }
}

mod tests {

    use super::*;
    use rayon::prelude::*;

    #[test]
    fn regression_test_main() {
        let args: Vec<&str> = vec![
            "DP1", // "DP1_5c_suggestions_off",
            "DP14",
        ];
        let mut heuristics = Heuristics {
            strategy: vec![0],
            max_coefficient: 1,
        };

        let mut expected_lengths = vec![
            1, //91,
            1,
        ];

        assert_eq!(
            args.len(),
            expected_lengths.len(),
            "Une longueur attendue par fichier de test"
        );
        args.par_iter().enumerate().for_each(|(i, file)| {
            let data =
                read(&"tests/", file).unwrap_or_else(|_| panic!("Erreur d'entrée pour {}", file));

            let axes = build_axes(&data, &heuristics);

            let mut partsols = vec![PartSol(Vec::new())];
            for candidate in data.candidates {
                let delta = candidate.actual_error;
                partsols = partsols
                    .par_iter()
                    .map(|s| extensions(s, &axes, delta))
                    .flatten()
                    .collect();
            }
            let sols: Vec<Vec<Axis>> = partsols.iter().map(|s| s.restrict(&axes)).collect();

            assert_eq!(
                sols.len(),
                expected_lengths[i],
                "Régression pour {} (coeff max 1)",
                file
            );
        });

        heuristics.max_coefficient = 3;
        expected_lengths = vec![
            1, //190,
            1,
        ];

        assert_eq!(
            args.len(),
            expected_lengths.len(),
            "Une longueur attendue par fichier de test"
        );

        args.par_iter().enumerate().for_each(|(i, file)| {
            let data =
                read(&"tests/", file).unwrap_or_else(|_| panic!("Erreur d'entrée pour {}", file));
            let axes = build_axes(&data, &heuristics);

            let mut partsols = vec![PartSol(Vec::new())];
            for candidate in data.candidates {
                let delta = candidate.actual_error;
                partsols = partsols
                    .par_iter()
                    .map(|s| extensions(s, &axes, delta))
                    .flatten()
                    .collect();
            }
            let sols: Vec<Vec<Axis>> = partsols.iter().map(|s| s.restrict(&axes)).collect();

            assert_eq!(
                sols.len(),
                expected_lengths[i],
                "Régression pour {} (coeff max 3)",
                file
            );
        });
    }
}
