//! Fonctions de base pour la lecture des fichiers de données et
//! pour l'écriture de fichiers de résultats.

use super::prelude::*;

use std::fs::File;
use std::path::*;
use std::{io, io::prelude::*};

use itertools::Itertools;

/// Structure de données auxiliaire, type d'entrée de *read_data*.
type DataHeaders = (
    char,
    Option<u16>,
    Vec<Candidate>,
    io::Lines<io::BufReader<File>>,
);

/// Lit les deux premières lignes du fichier d'entrée.
fn read_headers(f: File) -> Result<DataHeaders, io::Error> {
    let mut candidates = Vec::new();

    let mut lines = io::BufReader::new(f).lines();

    let line0 = lines.next().expect("Le fichier d'entrée semble vide.")?;
    let sep = line0.chars().next().unwrap();

    let total = match line0.split(sep).nth(2).unwrap().parse::<u16>() {
        Ok(t) => Some(t),
        _ => None,
    };

    for id in line0.split(sep).skip(3) {
        candidates.push(Candidate::new(String::from(id)));
    }

    let line1 = lines.next().expect("Erreur ligne 2 du fichier d'entrée.")?;
    for (i, s) in line1.split(sep).skip(3).enumerate() {
        candidates[i].actual_score = s
            .parse()
            .unwrap_or_else(|_| panic!("Erreur de score pour le candidat C{} ?", i + 1));
    }

    Ok((sep, total, candidates, lines))
}

/// Lit les lignes du fichier d'entrée à partir de la troisième.
fn read_data(data_headers: DataHeaders) -> Result<Data, io::Error> {
    let mut statuses = Vec::<Status>::new();
    let mut suggestions = Vec::<Option<bool>>::new();
    let (sep, total, mut candidates, lines) = data_headers;
    let mut answer_buffer: Vec<Vec<bool>> =
        candidates.iter().map(|_| Vec::with_capacity(5)).collect();

    for (i, l) in lines.enumerate() {
        let u = l.unwrap();
        let mut cells = u.split(sep);
        let a = cells.next().unwrap();
        if i % 5 == 1 {
            statuses.push(a.parse().unwrap());
        }

        let suggestion = match cells.nth(1).unwrap() {
            "" => Some(false),
            "1" => Some(true),
            "?" => None,
            x => panic!("Erreur ligne {} du fichier d'entrée : {}", i + 3, x),
        };
        suggestions.push(suggestion);

        for (j, a) in cells.enumerate() {
            answer_buffer[j].push(a != "");
            if i % 5 == 4 {
                candidates[j].answers.push(answer_buffer[j].clone());
                answer_buffer[j] = Vec::with_capacity(5);
            }
        }
    }

    let maximal_score = match total {
        Some(t) => 10 * t,
        None => 10 * (statuses.len() as u16),
    };

    for mut c in &mut candidates {
        c.actual_error = maximal_score - c.actual_score;
    }

    let deltas = Errors(candidates.iter().map(|c| c.actual_error).collect());

    Ok(Data {
        statuses,
        suggestions,
        candidates,
        deltas,
    })
}

/// Lit dans le fichier d'entrée les statuts des questions, et les scores et
/// les réponses des candidats.
pub fn read(path: &str, filename: &str) -> Result<Data, io::Error> {
    let f = File::open(&Path::new(path).join(format!("{}.tsv", filename))).unwrap_or_else(|_| {
        File::open(&Path::new(path).join(format!("{}.csv", filename)))
            .expect(&format!("Fichier d'entrée manquant : {}", filename))
    });
    let data_headers = read_headers(f)?;
    read_data(data_headers)
}

/// Exporte dans le fichier de sortie les différentes corrections calculées.
pub fn write(sols: &[Vec<Axis>], f: &Path) -> Result<(), io::Error> {
    if sols.is_empty() {
        return Ok(());
    }

    let all_coeffs_are_trivial: bool = sols
        .iter()
        .all(|s| s.iter().all(|a| a.iter().all(|q| q.coeff == Coeff::A)));

    let first = sols.first().unwrap();
    let n = first.len();

    let is_common: Vec<bool> = (0..n)
        .map(|i| sols.iter().all(|s| s[i] == first[i]))
        .collect();

    let mut matrix = Vec::<Vec<String>>::new();

    // écriture des deux premières lignes
    let mut block = Vec::<String>::new();
    block.resize(5 * n + 1, String::new());
    for i in 0..n {
        block[5 * i + 1] = format!("Q{}", i + 1);
    }
    matrix.push(block.clone());

    for i in 0..n {
        for (j, c) in "ABCDE".chars().enumerate() {
            block[5 * i + j + 1] = c.to_string();
        }
    }
    matrix.push(block);

    let mut coeffs = vec![String::from(""); 5];

    // écriture des axes communs
    let max_axes: usize = (0..n)
        .filter(|i| is_common[*i])
        .map(|i| first[i].len())
        .max()
        .unwrap_or(0);
    for k in 0..max_axes {
        let mut block_coeffs = vec![String::from("S*")];
        let mut block = vec![String::new()];
        for i in 0..n {
            match (is_common[i], first[i].get(k)) {
                (true, Some(q)) => {
                    block.extend(q.to_str_vec());
                    coeffs[2] = q.coeff.to_string();
                    block_coeffs.extend(coeffs.clone())
                }
                _ => {
                    block.extend(vec![String::from(""); 5]);
                    block_coeffs.extend(vec![String::from(""); 5])
                }
            };
        }

        if !all_coeffs_are_trivial {
            matrix.push(block_coeffs);
        }
        matrix.push(block);
    }

    // écriture des quintuplets spécifiques
    for (k, sol) in sols.iter().enumerate() {
        let max_axes: usize = (0..n)
            .filter(|i| !is_common[*i])
            .map(|i| sol[i].len())
            .max()
            .unwrap_or(0);
        for l in 0..max_axes {
            let mut block_coeffs = vec![format!("S{}", k + 1)];
            let mut block = vec![String::new()];
            for i in 0..n {
                match (is_common[i], sol[i].get(l)) {
                    (false, Some(q)) => {
                        block.extend(q.to_str_vec());
                        coeffs[2] = q.coeff.to_string();
                        block_coeffs.extend(coeffs.clone())
                    }
                    _ => {
                        block.extend(vec![String::from(""); 5]);
                        block_coeffs.extend(vec![String::from(""); 5])
                    }
                }
            }
            if !all_coeffs_are_trivial {
                matrix.push(block_coeffs.clone());
            }
            matrix.push(block.clone());
        }
    }

    // export dans le fichier de sortie
    let file = File::create(f).unwrap_or_else(|_| panic!("Fichier de sortie non créé : {:?}", f));

    for j in 0..5 * n + 1 {
        writeln!(
            &file,
            "{}",
            matrix
                .iter()
                .map(|row| row[j].clone())
                .join(&String::from("\t"))
        )
        .unwrap();
    }
    Ok(())
}

/// Lit le fichier d'heuristiques, en tire les deltas et la stratégie choisis.
pub fn read_heuristics(path: &Path, filename: &Path) -> Result<Heuristics, io::Error> {
    let f = File::open(path.join(filename)).expect("Fichier d'heuristiques manquant ?");
    let mut lines = io::BufReader::new(f)
        .lines()
        .map(|l| l.expect("Erreur dans les lignes du fichier d'heuristiques"))
        .filter(|l| !l.starts_with('#') || l.chars().next() == None);

    for _ in 0..2 {
        lines.next();
    }

    let mut heuristics = Heuristics::new();

    match lines.next() {
        None => panic!("Ligne absente ou modifiée dans le fichier d'heuristiques ?"),
        Some(h) => {
            let parts: Vec<_> = h.split('=').map(|s| s.trim()).collect();
            if parts.len() != 2 {
                panic!("Champ manquant dans le fichier d'heuristiques ? {}", h);
            }

            let b: &[_] = &['[', ']'];
            let field: Vec<_> = parts[1]
                .trim_matches(b)
                .split(',')
                .map(|s| s.trim())
                .collect();
            for v in field {
                heuristics.strategy.push(
                    v.parse()
                        .unwrap_or_else(|_| panic!("Erreur de parsing : {}", h)),
                );
            }
        }
    }

    if let Some(h) = lines.next() {
        let parts: Vec<_> = h.split('=').map(|s| s.trim()).collect();
        if parts.len() == 1 {
            panic!("Champ manquant dans le fichier d'heuristiques ? {}", h);
        }
        let field: u16 = parts[1]
            .parse()
            .unwrap_or_else(|_| panic!("Erreur de parsing : {}", h));
        heuristics.max_coefficient = field
    }

    Ok(heuristics)
}

#[cfg(test)]
mod tests {

    use super::*;

    #[test]
    fn read_total_works() {
        match read_headers(File::open(&Path::new("tests").join(&Path::new("DP10.csv"))).unwrap()) {
            Err(a) => panic!("{:?}", a),
            Ok((_, total, _, _)) => assert_eq!(total, Some(18)),
        }

        match read_headers(File::open(&Path::new("tests").join(&Path::new("DP1.csv"))).unwrap()) {
            Err(a) => panic!("{:?}", a),
            Ok((_, total, _, _)) => assert_eq!(total, None),
        }
    }

    #[test]
    fn read_headers_works() {
        match read_headers(File::open(&Path::new("tests").join(&Path::new("DP1.csv"))).unwrap()) {
            Err(a) => panic!("{:?}", a),
            Ok((_, _, candidates, _)) => {
                assert_eq!(candidates.len(), 14);

                let c = &candidates[0];
                assert_eq!(c.id, "C1");
                assert_eq!(c.actual_score, 142);

                let c = &candidates[1];
                assert_eq!(c.id, "C2");
                assert_eq!(c.actual_score, 145);

                let c = &candidates[13];
                assert_eq!(c.id, "C14");
                assert_eq!(c.actual_score, 82);
            }
        }
    }

    #[test]
    fn read_with_coeffs_works() {
        match read(&"tests", &"DP10") {
            Err(a) => panic!("{:?}", a),
            Ok(d) => {
                assert_eq!(d.candidates[0].actual_error, 23);
                assert_eq!(d.candidates[1].actual_error, 33);
                assert_eq!(d.candidates[3].actual_error, 38);
            }
        }
    }

    #[test]
    fn read_works() {
        match read(&"tests", &"DP1") {
            Err(a) => panic!("{:?}", a),
            Ok(d) => {
                assert_eq!(d.statuses.len(), 15);
                assert_eq!(d.statuses[0], Status::QCM);
                assert_eq!(d.statuses[4], Status::QRU);
                assert_eq!(d.statuses[14], Status::QCM);

                assert_eq!(d.candidates[0].answers[0][1], true);

                assert_eq!(d.candidates[0].actual_error, 8);
                assert_eq!(d.candidates[1].actual_error, 5);
                assert_eq!(d.candidates[6].answers[6][3], false);

                assert_eq!(d.candidates[13].answers[10][3], true);
            }
        }

        match read(&"tests", &"DP14") {
            Err(a) => panic!("{:?}", a),
            Ok(d) => {
                assert_eq!(d.statuses.len(), 15);
                assert_eq!(d.statuses[0], Status::QRU);
                assert_eq!(d.statuses[1], Status::QCM);
            }
        }
    }

    #[test]
    fn read_heuristics_works() {
        match read_heuristics(&Path::new("tests"), &Path::new("heuristiques.toml")) {
            Err(a) => panic!("{:?}", a),
            Ok(h) => {
                assert_eq!(h.strategy, vec![0]);
            }
        }

        match read_heuristics(
            &Path::new("tests"),
            &Path::new("heuristiques_enrichies.toml"),
        ) {
            Err(a) => panic!("{:?}", a),
            Ok(h) => {
                assert_eq!(h.strategy, vec![0, 1, 5]);
            }
        }
    }
}
