//! Structures de données élémentaires.

use std::cmp::{Ordering, PartialOrd};
use std::convert::Into;
use std::fmt;
use std::ops::*;
use std::str::FromStr;

use itertools::Itertools;

//----------------------------------------------------------------------------

/// Représente les cinq valeurs possibles de la correction d'une proposition.
#[derive(Debug, PartialEq, Eq, PartialOrd, Ord, Clone)]
pub enum Answer {
    MZ,
    F,
    ANN,
    V,
    PMZ,
}

/// Représente les deux statuts possibles d'une question (QCM ou QRU).
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum Status {
    QRU,
    QCM,
}

impl FromStr for Status {
    type Err = ();

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            "QCM" => Ok(Self::QCM),
            "QRU" => Ok(Self::QRU),
            _ => Err(()),
        }
    }
}

/// Représente les trois coefficients possibles d'une question.
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum Coeff {
    A,
    B,
    C,
}

impl Into<u16> for Coeff {
    fn into(self) -> u16 {
        match self {
            Coeff::A => 1,
            Coeff::B => 2,
            Coeff::C => 3,
        }
    }
}

impl fmt::Display for Coeff {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let c: u16 = self.clone().into();
        write!(f, "{}", c)
    }
}

//----------------------------------------------------------------------------

/// Représente (la copie d')un candidat individuel. Les réponses sont groupées
/// par paquets de cinq.
#[derive(Clone, Debug)]
pub struct Candidate {
    pub id: String,
    pub actual_score: u16,
    pub actual_error: u16,
    pub answers: Vec<Vec<bool>>,
}

impl Candidate {
    pub fn new(id: String) -> Self {
        Self {
            id,
            actual_score: 0,
            actual_error: 0,
            answers: Vec::new(),
        }
    }
}

/// Calcule l'erreur locale d'un candidat donné pour une question donnée.
pub fn local_error(answers: &[bool], corrs: &[Answer], s: Status) -> u16 {
    let mut empty = true;
    let mut discordances = 0;
    for (answer, corr) in answers.iter().zip(corrs.iter()) {
        empty &= !answer;
        match (answer, corr, s) {
            (true, Answer::MZ, _) => return 10,
            (false, Answer::PMZ, _) => return 10,
            (true, Answer::F, Status::QRU) => return 10,
            (false, Answer::V, Status::QRU) => return 10,
            (true, Answer::F, Status::QCM) => discordances += 1,
            (false, Answer::V, Status::QCM) => discordances += 1,
            _ => (),
        }
        if discordances > 2 {
            return 10;
        }
    }
    if empty {
        return 10;
    }
    match discordances {
        0 => 0,
        1 => 5,
        _ => 8,
    }
}

//----------------------------------------------------------------------------

/// Structure correspondant à une liste d'erreurs, pour plusieurs candidats en
/// même temps, pour une ou plusieurs questions.
#[derive(Clone, Debug, PartialEq, Eq, Ord, Hash)]
pub struct Errors(pub Vec<u16>);

impl AddAssign for Errors {
    /// Addition composante par composante.
    fn add_assign(&mut self, rhs: Self) {
        self.0
            .iter_mut()
            .zip(rhs.0.iter())
            .map(|(a, b)| *a += b)
            .collect()
    }
}

impl Add for Errors {
    type Output = Self;
    /// Addition composante par composante.
    fn add(self, rhs: Self) -> Self {
        Self(
            self.0
                .iter()
                .zip(rhs.0.iter())
                .map(|(a, b)| a + b)
                .collect(),
        )
    }
}

impl PartialOrd for Errors {
    /// Comparaison globale composante par composante.
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        if let Some((i, (l, r))) = self
            .0
            .iter()
            .zip(other.0.iter())
            .enumerate()
            .find(|(_, (u, v))| u != v)
        {
            if l < r
                && self.0[i + 1..]
                    .iter()
                    .zip(other.0[i + 1..].iter())
                    .all(|(a, b)| a <= b)
            {
                Some(Ordering::Less)
            } else if l > r
                && self.0[i + 1..]
                    .iter()
                    .zip(other.0[i + 1..].iter())
                    .all(|(a, b)| a >= b)
            {
                Some(Ordering::Greater)
            } else {
                None
            }
        } else {
            Some(Ordering::Equal)
        }
    }
}

impl Index<usize> for Errors {
    type Output = u16;
    fn index(&self, j: usize) -> &u16 {
        &self.0[j]
    }
}

//----------------------------------------------------------------------------

/// Structure correspondant à la tranche d'un corrigé pour une question fixée,
/// en mémoïsant l'erreur locale de tous les candidats.
#[derive(Clone, Debug, PartialEq, Eq)]
pub struct Quintuple {
    pub corrs: Vec<Answer>, // les réponses proposées
    pub errors: Errors,     // l'erreur attribuée à chaque candidat
    pub coeff: Coeff,       // le coefficient retenu (dans {1, 2, 3})
}

impl Quintuple {
    pub fn new(answers: &[Vec<bool>], corrs: &[Answer], s: Status, c: u16) -> Self {
        Self {
            corrs: corrs.to_vec(),
            errors: Errors(
                answers
                    .iter()
                    .map(|a| c * local_error(a, corrs, s))
                    .collect(),
            ),
            coeff: match c {
                1 => Coeff::A,
                2 => Coeff::B,
                _ => Coeff::C,
            },
        }
    }

    pub fn to_str_vec(&self) -> Vec<String> {
        self.corrs.iter().map(|a| format!("{:?}", a)).collect()
    }
}

/// Collection de quintuplets admissibles correspondant à une question fixée.
/// L'espace de recherche sera un produit cartésien (filtré) d'axes.
pub type Axis = Vec<Quintuple>;

//----------------------------------------------------------------------------

/// Structure de données correspondant au contenu d'un fichier de données.
#[derive(Clone)]
pub struct Data {
    pub statuses: Vec<Status>,
    pub suggestions: Vec<Option<bool>>,
    pub candidates: Vec<Candidate>,
    pub deltas: Errors,
}

/// Structure de données correspondant au contenu d'un fichier d'heuristiques.
pub struct Heuristics {
    pub strategy: Vec<usize>,
    pub max_coefficient: u16,
}

impl Heuristics {
    pub fn new() -> Self {
        Self {
            strategy: Vec::new(),
            max_coefficient: 1,
        }
    }
}

//----------------------------------------------------------------------------

/// Structure correspondant à une (sous-)partition d'un entier ; on mémoïse la
/// somme des parties, ainsi que l'entier _aim_ que l'on cherche à partitionner.
/// Par exemple, vec![(0, 2), (2, 1), (3, 0)] peut être vu comme une
/// sous-partition de somme 3, de domaine {0, 2, 3}, obtenue comme résultat
/// intermédiaire dans l'énumération des partitions de 5, de domaine {0, ..., 3}.
#[derive(Clone, Debug)]
pub struct Partition {
    pub parts: Vec<(usize, u16)>,
    pub sum: u16,
    pub aim: u16,
}

impl Partition {
    /// La sous-partition vide (sans aucune partie) de l'entier _aim_.
    fn empty(aim: u16) -> Self {
        Self {
            parts: Vec::new(),
            sum: 0,
            aim,
        }
    }

    /// La sous-partition ayant une seule partie, de position donnée.
    pub fn new(aim: u16, pos: usize, part: u16) -> Self {
        Self {
            parts: vec![(pos, part)],
            sum: part,
            aim,
        }
    }

    /// Conversion d'une partition supposée complète en vecteur de sommants.
    pub fn to_vec(&self) -> Vec<u16> {
        let max_pos = self.parts.iter().map(|(pos, _)| pos).max().unwrap();
        let mut result = vec![0; 1 + max_pos];
        for (pos, part) in &self.parts {
            result[*pos] = *part;
        }
        result
    }
}

impl AddAssign for Partition {
    fn add_assign(&mut self, rhs: Self) {
        self.parts.extend(rhs.parts);
        self.sum += rhs.sum;
    }
}

impl Add for Partition {
    type Output = Self;
    fn add(self, rhs: Self) -> Self {
        let mut aux = self;
        aux += rhs;
        aux
    }
}

/// Structure correspondant à un ensemble de (sous-)partitions.
#[derive(Clone, Debug)]
pub struct PartSet(pub Vec<Partition>);

/// Deux ensembles de sous-partitions correspondant à des parties couvrant des
/// domaines disjoints peuvent être fusionnés en un ensemble de sous-partitions
/// couvrant la réunion des deux domaines.
impl Mul for PartSet {
    type Output = Self;

    fn mul(self, other: Self) -> Self {
        Self(
            self.0
                .iter()
                .cartesian_product(other.0.iter())
                .map(|(p, q)| p.clone().add(q.clone()))
                .filter(|p| p.sum <= p.aim)
                .collect(),
        )
    }
}

impl PartSet {
    pub fn new(aim: u16) -> Self {
        Self(vec![Partition::empty(aim)])
    }
}

//----------------------------------------------------------------------------

#[cfg(test)]
mod tests {

    use super::*;

    #[test]
    fn order_on_answers_works() {
        assert!(Answer::ANN < Answer::PMZ);
        assert!(Answer::V > Answer::ANN);
        assert!(Answer::V > Answer::F);
    }

    #[test]
    fn new_candidate_works() {
        let mut c = Candidate::new(String::from("C1"));
        c.actual_score = 135;
    }

    #[test]
    fn local_error_works() {
        assert_eq!(
            local_error(
                &[true; 5],
                &[Answer::V, Answer::V, Answer::F, Answer::V, Answer::F,],
                Status::QCM
            ),
            8
        );
    }

    #[test]
    fn partition_to_vec_works() {
        let aim = 37;
        let mut p = Partition::empty(aim);
        for (pos, part) in &[(3, 22), (2, 0), (4, 35), (0, 11), (1, 7)] {
            p += Partition::new(aim, *pos, *part);
        }
        let mut parts = p.parts.clone();
        parts.sort_by_key(|&(pos, _)| pos);

        assert_eq!(
            p.to_vec(),
            parts
                .iter()
                .cloned()
                .enumerate()
                .map(|(i, (pos, part))| {
                    assert_eq!(i, pos);
                    part
                })
                .collect::<Vec<_>>()
        );
    }

    #[test]
    fn order_on_errors_work() {
        let left = Errors(vec![0, 8, 0, 0, 0, 8, 8, 5, 0, 5, 0, 5, 8, 8]);
        let right = Errors(vec![5, 8, 10, 10, 13, 18, 23, 25, 25, 33, 35, 38, 46, 68]);
        assert!(left <= right);
    }
}
