//! Fonctions principales permettant de parcourir les solutions.

use super::prelude::*;

use itertools::{multizip, Itertools};
use rayon::iter::{IntoParallelRefIterator, ParallelIterator};
use rayon::prelude::*;

use std::collections::HashSet;

/// Engendre itérativement les éléments du produit cartésien des termes.
/// Loué soit Stargateur : <https://stackoverflow.com/a/57517984>.
pub fn product<T: Clone>(terms: &[Vec<T>]) -> impl Iterator<Item = Vec<T>> + '_ {
    terms
        .iter()
        .map(|x| x.iter().cloned())
        .multi_cartesian_product()
}

/// Récupère les données extraites du fichier d'entrée, et construit une liste
/// de domaines possibles pour chacune des questions. L'espace de recherche
/// sera un produit cartésien (filtré) de ces domaines.
pub fn build_axes(d: &Data, h: &Heuristics) -> Vec<Axis> {
    let mut consensus = Vec::<Option<bool>>::new();
    for (i, _) in d.suggestions.iter().enumerate() {
        let mut answers = d.candidates.iter().map(|c| c.answers[i / 5][i % 5]);
        let init = answers.next().unwrap();
        consensus.push(answers.fold(Some(init), |acc, ans| match (acc, ans) {
            (None, _) => None,
            (Some(true), false) => None,
            (Some(false), true) => None,
            _ => acc,
        }));
    }

    let mut result = Vec::new();

    for (sugg, cons, (i, stat)) in multizip((
        d.suggestions.chunks(5),
        consensus.chunks(5),
        d.statuses.iter().enumerate(),
    )) {
        result.push(compatible_quintuples(
            DataQuints {
                suggestions: sugg,
                consensus: cons,
                answers: d
                    .candidates
                    .iter()
                    .cloned()
                    .map(|c| c.answers[i].clone())
                    .collect::<Vec<_>>()
                    .as_slice(),
                status: *stat,
                deltas: d.deltas.clone(),
            },
            h,
        ));
    }
    result
}

/// Structure de données auxiliaire, type d'entrée de _compatible_quintuples_,
/// associée à une question fixée.
/// Le champ _answers_ correspond aux réponses des candidats à cette question,
/// le champ _deltas_ aux erreurs globales de l'ensemble des candidats.
#[derive(Debug)]
struct DataQuints<'a> {
    suggestions: &'a [Option<bool>],
    consensus: &'a [Option<bool>],
    answers: &'a [Vec<bool>],
    status: Status,
    deltas: Errors,
}

/// Construit, pour une question donnée, un vecteur contenant tous les
/// quintuplets de réponses compatibles avec les suggestions,
/// en réduisant l'espace de recherche par exploitation des consensus.
/// On filtre le cas échéant en appliquant les stratégies retenues.
fn compatible_quintuples(d: DataQuints<'_>, h: &Heuristics) -> Axis {
    let mut opts = Vec::<Vec<Answer>>::with_capacity(5);
    let s = d.status;

    for (suggestion, consensus) in d.suggestions.iter().zip(d.consensus.iter()) {
        // Contraintes à l'échelle des propositions

        // Spécifications issues des consensus
        let mut v = match consensus {
            Some(true) => vec![Answer::V, Answer::F, Answer::MZ],
            Some(false) => vec![Answer::V, Answer::F, Answer::PMZ],
            _ => vec![Answer::V, Answer::F, Answer::PMZ, Answer::ANN, Answer::MZ],
        };

        // Spécifications issues du statut de la question
        if s == Status::QRU {
            v = v
                .iter()
                .cloned()
                .filter(|ans| ![Answer::MZ, Answer::PMZ].contains(ans))
                .collect();
        }

        // Spécifications issues des suggestions
        v = match suggestion {
            Some(true) => v
                .iter()
                .cloned()
                .filter(|ans| ans >= &Answer::ANN)
                .collect(),
            Some(false) => v
                .iter()
                .cloned()
                .filter(|ans| ans <= &Answer::ANN)
                .collect(),
            None => v,
        };

        // Spécifications issues des choix stratégiques
        if h.strategy == vec![0] {
            v = v
                .iter()
                .cloned()
                .filter(|ans| ans != &Answer::ANN)
                .collect();
        };

        opts.push(v);
    }

    let mut result: Vec<_> = product(&opts)
        .cartesian_product(1..=h.max_coefficient)
        .map(|(v, c)| Quintuple::new(d.answers, v.as_slice(), d.status, c))
        .filter(|q| q.errors <= d.deltas)
        .collect();

    // Contraintes à l'échelle de la question

    if s == Status::QRU {
        result = result
            .iter()
            .cloned()
            .filter(|q| q.corrs.iter().filter(|&ans| ans == &Answer::V).count() == 1)
            .collect();
    }

    result = result
        .iter()
        .cloned()
        .filter(|q| {
            h.strategy
                .contains(&q.corrs.iter().filter(|&ans| ans == &Answer::ANN).count())
        })
        .collect();

    result
}

//----------------------------------------------------------------------------

/// Construit l'ensemble des sous-partitions de delta correspondant à un axe
/// situé en position _pos_ dans la liste des questions, pour le candidat n°j.
fn subpartitions(pos: usize, axis: &[Quintuple], delta: u16, j: usize) -> PartSet {
    let mut errors: HashSet<u16> = HashSet::with_capacity(9);

    for e in axis.iter().map(|q| q.errors[j]) {
        errors.insert(e);
        if errors.len() == 9 {
            break;
        }
    }

    PartSet(
        errors
            .iter()
            .map(|e| Partition::new(delta, pos, *e))
            .collect(),
    )
}

//----------------------------------------------------------------------------

/// Structure auxiliaire correspondant à une liste de partitions associée à un
/// segment initial de l'ensemble des candidats.
#[derive(Clone)]
pub struct PartSol(pub Vec<Vec<u16>>);

impl PartSol {
    /// Restreint les axes de manière à ce que l'erreur attribuée aux premiers
    /// candidats corresponde à celle fixée par les partitions.
    pub fn restrict(&self, axes: &[Axis]) -> Vec<Axis> {
        axes.iter()
            .cloned()
            .enumerate()
            .map(|(i, axis)| {
                axis.par_iter()
                    .cloned()
                    .filter(|quintuple| {
                        self.0
                            .iter()
                            .enumerate()
                            .all(|(j, p)| quintuple.errors[j] == p[i])
                    })
                    .collect()
            })
            .collect()
    }

    fn push(&mut self, part: Partition) {
        self.0.push(part.to_vec())
    }
}

/// Calcule les extensions possibles de la solution partielle passée en argument,
/// en listant les partitions possibles de l'erreur globale de la copie suivante.
pub fn extensions(sol: &PartSol, axes: &[Axis], delta: u16) -> Vec<PartSol> {
    let j = sol.0.len();
    sol.restrict(axes)
        .par_iter()
        .enumerate()
        .map(|(pos, axis)| subpartitions(pos, axis, delta, j))
        .reduce(|| PartSet::new(delta), |u, v| u * v)
        .0
        .par_iter()
        .filter(|p| p.sum == p.aim)
        .cloned()
        .map(|p| {
            let mut w = sol.clone();
            w.push(p);
            w
        })
        .collect()
}

#[cfg(test)]
mod tests {

    use super::super::file_io::*;
    use super::*;
    use std::path::Path;

    #[test]
    fn product_works() {
        let v1 = vec![0, 1];
        let v2 = vec![2, 3];
        let c = &vec![v1, v2];
        let p: Vec<_> = product(c).collect();
        assert_eq!(p, vec![&[0, 2], &[0, 3], &[1, 2], &[1, 3]]);
    }

    #[test]
    fn buildaxes_works() {
        let data = read(&Path::new("tests").join(Path::new("DP14.csv"))).expect("");

        let h = Heuristics {
            strategy: vec![0],
            max_coefficient: 1,
        };

        let axis = &build_axes(&data, &h)[1];
        assert_eq!(axis.len(), 16);
        assert_eq!(
            axis.iter().map(|q| q.errors.0[0]).collect::<Vec<_>>(),
            vec![8, 10, 8, 10, 10, 10, 10, 10, 8, 10, 8, 10, 10, 10, 10, 10]
        );

        let data = read(&Path::new("tests").join(Path::new("DP1.csv"))).expect("");
        assert_eq!(data.candidates[0].actual_error, 8);
        assert_eq!(data.candidates[1].actual_error, 5);

        let all_axes = build_axes(&data, &h);
        assert_eq!(all_axes.len(), 15);

        let axis = all_axes[14].clone();
        assert!(axis
            .iter()
            .cloned()
            .map(|q| q.corrs)
            .any(|x| x == vec![Answer::F, Answer::V, Answer::F, Answer::F, Answer::V]));

        let data = read(&Path::new("tests").join(Path::new("DP1_swap.csv"))).expect("");
        assert_eq!(data.candidates[0].actual_error, 5);
        assert_eq!(data.candidates[1].actual_error, 8);

        let all_axes = build_axes(&data, &h);
        assert_eq!(all_axes.len(), 15);

        let axis = all_axes[14].clone();
        assert!(axis
            .iter()
            .cloned()
            .map(|q| q.corrs)
            .any(|x| x == vec![Answer::F, Answer::V, Answer::F, Answer::F, Answer::V]));
    }

    #[test]
    fn axes_non_increasing_works() {
        let data_on =
            read(&Path::new("tests").join(Path::new("DP1_5c_suggestions_on.csv"))).expect("");
        let data_off =
            read(&Path::new("tests").join(Path::new("DP1_5c_suggestions_off.csv"))).expect("");

        let mut h = Heuristics {
            strategy: vec![0],
            max_coefficient: 1,
        };

        let mut results = Vec::<Vec<usize>>::new();

        for d in &[data_on, data_off] {
            for i in 1..4 {
                h.max_coefficient = i;
                results.push(build_axes(d, &h).iter().map(|a| a.len()).collect());
            }
        }

        // croissance pour chaque état de suggestion
        for j in 0..2 {
            for i in 0..2 {
                for k in 0..14 {
                    assert!(results[3 * j + i][k] <= results[3 * j + i + 1][k]);
                }
            }
        }

        // croissance entre les deux suggestions
        for i in 0..3 {
            for k in 0..14 {
                assert!(results[i][k] <= results[i + 3][k]);
            }
        }

        // TODO fix : `move` dans la closure ?
    }
}
