use core::{fmt, mem};

use crate::header::{Header, HeaderError};
use thiserror::Error;

#[derive(Debug, Error)]
pub enum ElfError {
    #[error(transparent)]
    Header(#[from] HeaderError),
}

type Result<T> = core::result::Result<T, ElfError>;

pub struct Elf<'a> {
    pub bytes: &'a [u8],
    pub header: &'a Header,
}

impl<'a> Elf<'a> {
    pub fn new(bytes: &'a [u8]) -> Self {
        let header: &'a Header = unsafe {
            mem::transmute(
                &*(&bytes[..mem::size_of::<Header>()] as *const [u8] as *const u8 as *const Header),
            )
        };

        Self { bytes, header }
    }

    pub fn try_new(bytes: &'a [u8]) -> Result<Self> {
        let elf = Self::new(bytes);
        elf.header.sanity_check()?;
        Ok(elf)
    }
}

impl<'a> fmt::Debug for Elf<'a> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{:?}", self.header)
    }
}
