extern crate edn_rs;

#[cfg(feature = "json")]
mod tests {
    use crate::edn_rs::json_to_edn;

    #[test]
    fn emits_helloworld_edn() {
        let json = String::from("{\"hello\": \"world\"}");
        let edn = String::from("{:hello \"world\"}");

        assert_eq!(edn, json_to_edn(json));
    }

    #[test]
    fn replaces_whitespace_in_keys_by_dash() {
        let json = String::from("{\"hello world\": \"julia\"}");
        let edn = String::from("{:hello-world \"julia\"}");

        assert_eq!(edn, json_to_edn(json));
    }

    #[test]
    fn emits_nil_edn() {
        let json = String::from("{\"hello\": null}");
        let edn = String::from("{:hello nil}");

        assert_eq!(edn, json_to_edn(json));
    }

    #[test]
    fn emits_char_edn() {
        let json = String::from("{\"hello\": 'c'}");
        let edn = String::from("{:hello \\c}");

        assert_eq!(edn, json_to_edn(json));
    }

    #[test]
    fn emits_number_edn() {
        let json = String::from("{\"multi_string with underscore\": 545643}");
        let edn = String::from("{:multi-string-with-underscore 545643}");

        assert_eq!(edn, json_to_edn(json));
    }

    #[test]
    fn multiline_json_to_edn() {
        let json = String::from(
            "{
            \"hello\": [
                {
                    \"country name\": \"brazil\",
                    \"word\": \"mundo\"
                },
                {
                    \"country name\": \"usa\",
                    \"word\": \"world\"
                }
            ]
        }",
        );
        let edn = String::from(
            "{
            :hello [
                {
                    :country-name \"brazil\",
                    :word \"mundo\"
                },
                {
                    :country-name \"usa\",
                    :word \"world\"
                }
            ]
        }",
        );

        assert_eq!(edn, json_to_edn(json));
    }

    #[test]
    fn regression_str_to_uint_test() {
        use edn_derive::Deserialize;
        use edn_rs::EdnError;
        #[derive(Deserialize, Debug, PartialEq)]
        struct A {
            amount: usize,
        }

        let a: Result<A, EdnError> = edn_rs::from_str("{ :amount \"123\" }");
        assert_eq!(a, Ok(A { amount: 123 }));
    }

    #[test]
    fn to_json() {
        use edn_rs::edn::{Edn, List, Map, Set, Vector};
        use edn_rs::{map, set};

        let edn = Edn::Vector(Vector::new(vec![
            Edn::Int(1),
            Edn::Double(1.2.into()),
            Edn::UInt(3),
            Edn::List(List::new(vec![
                Edn::Bool(false),
                Edn::Key(":f".to_string()),
                Edn::Nil,
                Edn::Rational("3/4".to_string()),
                Edn::Set(Set::new(set! {
                    Edn::Rational("3/4".to_string())
                })),
            ])),
            Edn::Map(Map::new(map![
                    String::from("false") => Edn::Key(":f".to_string()),
                    String::from("nil") => Edn::Rational("3/4".to_string()),
                    String::from(":my-crazy-map") => Edn::Map(Map::new(map![
                        String::from("false") => Edn::Map(
                            Map::new( map![
                                String::from(":f") => Edn::Key(String::from(":b"))
                            ])),
                        String::from("nil") => Edn::Vector(
                            Vector::new( vec![
                                Edn::Rational("3/4".to_string()),
                                Edn::Int(1isize)
                            ]))
                ]))
            ])),
        ]));

        assert_eq!(edn.to_json(),
            "[1, 1.2, 3, [false, \"f\", null, 0.75, [0.75]], {\"myCrazyMap\": {\"false\": {\"f\": \"b\"}, \"nil\": [0.75, 1]}, \"false\": \"f\", \"nil\": 0.75}]");
    }
}
