//! Read-only HTTP request input for HTTP-trigger serverless functions.
//!
//! For executing HTTP Requests, use [`crate::HttpFetch`].

use super::error::{Error, HTTPBodyTooLargeError, HeaderError, HttpError, UriError};
use http::Uri;
use http::{header::HeaderName, HeaderMap, Version};
use num_derive::FromPrimitive;
use num_traits::FromPrimitive;
use std::str;
use std::str::FromStr;
use std::{collections::HashMap, convert::TryFrom};
use url::form_urlencoded;

#[link(wasm_import_module = "request")]
extern "C" {
    fn http_cur_req_get_method() -> i32;
    fn http_cur_req_get_uri(err_code: *mut i32) -> i32;
    fn http_cur_req_get_body(error_code: *mut i32) -> i32;
    fn http_cur_req_get_all_header_keys(err_code: *mut i32) -> i32;
    fn http_cur_req_get_header_value(
        header_ptr: *const u8,
        header_len: i32,
        err_code: *mut i32,
    ) -> i32;
}

/// Request which may include version, body, headers, method, and URL.
///
#[derive(Debug)]
pub struct HttpRequest {
    version: Version,
    method: HttpMethod,
    uri: Uri,
    headers: HeaderMap,
    body: Option<Vec<u8>>,
}

/// Enum containing a number of common HTTP methods.
#[derive(Debug, FromPrimitive, PartialEq)]
pub enum HttpMethod {
    NONE = 0,
    GET = 1,
    HEAD = 2,
    POST = 3,
    PUT = 4,
    DELETE = 5,
    CONNECT = 6,
    OPTIONS = 7,
    TRACE = 8,
    PATCH = 9,
}

impl FromStr for HttpMethod {
    type Err = ();

    fn from_str(input: &str) -> Result<HttpMethod, Self::Err> {
        match input.to_uppercase().as_str() {
            "GET" => Ok(HttpMethod::GET),
            "HEAD" => Ok(HttpMethod::HEAD),
            "POST" => Ok(HttpMethod::POST),
            "PUT" => Ok(HttpMethod::PUT),
            "DELETE" => Ok(HttpMethod::DELETE),
            "CONNECT" => Ok(HttpMethod::CONNECT),
            "OPTIONS" => Ok(HttpMethod::OPTIONS),
            "TRACE" => Ok(HttpMethod::TRACE),
            "PATCH" => Ok(HttpMethod::PATCH),
            _ => Ok(HttpMethod::NONE),
        }
    }
}

impl HttpRequest {
    /// Returns the client request being handled by the serverless function.
    ///
    /// # Prefetch Body
    ///
    /// By passing the `prefetch_body` flag as true, the whole request body
    /// will be read from request at this function call time and then reside in the
    /// function memory.
    ///
    /// # Panics
    ///
    /// This method panics if the client request has already been retrieved by this method once and the body
    /// has been fetched.
    ///
    /// # Limits Exceeded Errors
    ///
    /// If the request exceeds the limits specified, this method sends an error response
    /// with a corresponding error code for the exceeding limit, so that the same can be
    /// handled explicitly by returning a customized error page.
    ///  
    ///             
    ///  
    /// Normally, this method does not need to be used explicity. EDJX sample
    /// [`Rust Serverless Application`](https://github.com/edjx/edjfunction-example-rust)
    /// code file (lib.rs) uses this method to fetch HTTP Request.
    ///
    /// `lib.rs`:
    ///
    /// ```
    /// #[no_mangle]
    /// pub fn init() {
    ///   let req = HttpRequest::from_client(true);
    ///   crate::serverless_function::serverless(req.unwrap());
    /// }
    /// ```
    ///
    /// `init()` function in `lib.rs` needs to be changed only if the HTTP body prefetch needs to be disabled or if a
    /// mutable reference to the request is required.
    ///
    pub fn from_client(prefetch_body: bool) -> Result<HttpRequest, HttpError> {
        let mut headers = HeaderMap::new();
        let body: Option<Vec<u8>>;
        let header_keys = get_all_header_keys_from_host()?;
        for key in header_keys {
            headers.insert(
                HeaderName::from_str(key.as_str()).unwrap(),
                get_header_value_from_host(key.as_str())?.parse().unwrap(),
            );
        }
        if prefetch_body {
            body = Some(get_body_from_host()?);
        } else {
            body = None;
        }
        Ok(HttpRequest {
            version: Version::HTTP_11,
            method: get_method_from_host(),
            uri: get_uri_from_host()?,
            headers: headers,
            body: body,
        })
    }

    /// Returns the enum value for HttpMethod.
    ///
    /// # Example
    ///
    /// ```no_run
    /// # use edjx::{HttpMethod, HttpRequest};
    /// use std::str::FromStr;
    ///
    /// fn serverless(req: &HttpRequest) {
    ///     let method = req.method();
    ///     assert_eq!(method, &HttpMethod::from_str("get").unwrap());
    /// }
    /// ```
    pub fn method(&self) -> &HttpMethod {
        return &self.method;
    }

    /// Returns the request URI value as [`http::Uri`] object. Use different functions available in 
    /// [`http::Uri`] for the corresponding
    /// object, as needed.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use edjx::{HttpRequest};
    ///
    /// fn serverless(req: &HttpRequest) {
    ///     let uri = req.uri();
    ///     assert_eq!(uri.path(), "/foo/bar");
    /// }
    /// ```
    pub fn uri(&self) -> &Uri {
        return &self.uri;
    }

    /// Returns the query string of request as an `Option` of [`&str`].
    pub fn query(&self) -> Option<&str> {
        return self.uri.query();
    }

    /// Returns the optional value for a provided key in query paramters.
    pub fn query_param_by_name(&self, name: &str) -> Option<String> {
        let query_params: HashMap<String, String> = self
            .uri
            .query()
            .map(|v| form_urlencoded::parse(v.as_bytes()).into_owned().collect())
            .unwrap_or_else(HashMap::new);

        match query_params.get(name) {
            Some(param_value) => Some(param_value.to_string()),
            None => None,
        }
    }

    /// Returns the path part of the request URI.
    pub fn path(&self) -> &str {
        return &self.uri.path();
    }

    /// Returns the map of headers in the request as an [`http::HeaderMap`] object.
    /// Use different functions available in [`http::HeaderMap`] for the corresponding
    /// object, as needed.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use edjx::{HttpRequest};
    ///
    /// fn serverless(req: &HttpRequest) {
    ///     let headers = req.headers();
    ///     assert!(headers.get("host").is_some());
    ///     assert_eq!(headers.get("host").unwrap(), &"example.com");
    /// }
    /// ```
    pub fn headers(&self) -> &HeaderMap {
        return &self.headers;
    }

    /// Returns the body associated with the request as a `Result` response of
    /// [`Vec<u8>`].
    ///
    /// # Prefetch Body
    ///
    /// If the HTTP request body was not prefetched, the same will be read on this method
    /// invocation.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use edjx::{error, HttpRequest, HttpResponse};
    ///
    /// fn serverless(req: &HttpRequest) {
    ///     let body = match req.body() {
    ///        Ok(body_vec_option) => body_vec_option,
    ///        Err(e) => {
    ///            error!("{}", &e.to_string());
    ///            HttpResponse::new()
    ///                .set_status(StatusCode::BAD_REQUEST)
    ///                .send()
    ///                .unwrap();
    ///            return;
    ///        }
    ///     }
    /// }
    /// ```
    pub fn body(&mut self) -> Result<&Option<Vec<u8>>, HttpError> {
        if self.body.is_none() {
            self.body = Some(get_body_from_host()?);
        }
        return Ok(&self.body);
    }
}

fn get_method_from_host() -> HttpMethod {
    let response = unsafe { http_cur_req_get_method() };

    let method_enum_value: Option<HttpMethod> = FromPrimitive::from_i32(response);
    match method_enum_value {
        Some(v) => v,
        None => HttpMethod::NONE,
    }
}

fn get_uri_from_host() -> Result<Uri, UriError> {
    let mut err = 0;
    let response = unsafe { http_cur_req_get_uri(&mut err) };

    if response > 0 {
        let value = String::from_utf8(super::result::get_result_bytes(response).unwrap()).unwrap();
        value.parse().map_err(|_| UriError::InvalidUri)
    } else {
        Err(UriError::try_from(err).unwrap())
    }
}

fn get_body_from_host() -> Result<Vec<u8>, HTTPBodyTooLargeError> {
    let mut err = 0;
    let response = unsafe { http_cur_req_get_body(&mut err) };

    if response > 0 {
        let result = super::result::get_result_bytes(response);
        Ok(result.unwrap())
    } else if response == 0 {
        Ok(Vec::with_capacity(0))
    } else {
        assert!(err == Error::HTTPTooLargeBody as i32);
        return Err(HTTPBodyTooLargeError());
    }
}

fn get_all_header_keys_from_host() -> Result<Vec<String>, HeaderError> {
    let mut err = 0;
    let response = unsafe { http_cur_req_get_all_header_keys(&mut err) };

    if response > 0 {
        let data = super::result::get_result_bytes(response).unwrap();

        let keys_delimited_string = str::from_utf8(&data).unwrap();
        let mut v = Vec::new();
        for header_key in keys_delimited_string.split(",") {
            if header_key.len() > 0 {
                v.push(String::from(header_key))
            }
        }
        Ok(v)
    } else if response == 0 {
        Ok(Vec::with_capacity(0))
    } else {
        assert!(err == Error::HTTPTooLargeHeaderName as i32);
        Err(HeaderError::TooLargeName)
    }
}

fn get_header_value_from_host(header_key: &str) -> Result<String, HeaderError> {
    let mut key = String::from(header_key);
    let key = key.as_mut_str();
    let dest_length = key.len() as i32;
    let dest_pointer = key.as_mut_ptr() as *const u8;
    let mut err = 0;

    let result_size = unsafe { http_cur_req_get_header_value(dest_pointer, dest_length, &mut err) };
    if result_size > 0 {
        Ok(String::from_utf8(super::result::get_result_bytes(result_size).unwrap()).unwrap())
    } else if result_size == 0 {
        Ok(String::with_capacity(0))
    } else {
        Err(HeaderError::from_i32(err).unwrap())
    }
}
