#![allow(dead_code)]
//! # EDJX Library SDK for Rust
//! This crate contains the EDJX Library SDK for Rust. This SDK is used to develop EDJX serverless functions.
//! In this version, only HTTP trigger functions are supported. That is, functions can be triggered using
//! HTTP only.
//!
//!
//! ## EDJX HTTP Trigger Functions
//! These functions are triggered from standard HTTP methods. The input
//! for the function is an HTTP Request object (represented by [`HttpRequest`] in the EDJX SDK)
//! and the return value has to be an HTTP Response object (represented by [`HttpResponse`] in the EDJX SDK).
//!
//! EDJX sample [`Rust Serverless Application`](https://github.com/edjx/edjfunction-example-rust) has two
//! Rust files :
//!
//! 1. `lib.rs` : This file contains helper code to fetch HTTP Requests and pass the request as input to the
//! serverless function being developed. Additionally, this file has code related to HTTP Response
//! processing. Modify this file only when the HTTP body prefetch
//! needs to be disabled for HTTP Request, a mutable reference to the request is required, or the
//! HTTP Response type needs to be changed. See [`HttpRequest`] and [`HttpResponse`] for more details.
//!
//! ```
//! #[no_mangle]
//! pub fn init() {
//!     let req = match HttpRequest::from_client(true) {
//!         Ok(val) => val,
//!         Err(e) => {
//!             error!("{}", e.to_string().as_str());
//!             HttpResponse::new()
//!                 .set_status(StatusCode::BAD_REQUEST)
//!                .send()
//!                .unwrap();
//!            return;
//!        }
//!    };
//!
//!   let res = crate::serverless_function::serverless(req);
//!     match res.send() {
//!         Ok(x) => x,
//!         Err(e) => {
//!             error!("{}", e.to_string().as_str());
//!         }
//!     };
//! }
//! ```
//!
//! 2. `serverless_function.rs` : This file contains the code to create a Serverless Function.
//!
//! ```
//! pub fn serverless(req: HttpRequest) -> HttpResponse {
//!     return HttpResponse::from("Success from EDJX".to_string())
//! }
//! ```
//!
//! [`HttpRequest`] is read-only, that is, it has only methods to read input HTTP Request.
//!  
//! Both [`HttpRequest`] and [`HttpResponse`] APIs are similar to [`standard RUST HTTP interface`](
//! https://docs.rs/http/0.2.4/http/). The other important HTTP structs like Headers,
//! Method, StatusCode, and Version are similar to [`standard RUST HTTP interface`](
//! https://docs.rs/http/0.2.4/http/).
//!
//!
//! Two other important types in this crate are [`HttpFetch`] and [`FetchResponse`]. [`HttpFetch`] is used
//! to execute HTTP Requests. [`FetchResponse`] is returned as a response for a Fetch Request. Both
//! these types APIs are similar to [`standrard RUST HTTP interface`](
//! https://docs.rs/http/0.2.4/http/)
//!
//! ### HTTP body size limits
//! Currently, all HTTP APIs have a size limit of 512 MB for the HTTP body.
//!
//!
//! ## Interacting with the EDJX Object Store and KV Store
//!
//! The most important APIs in the EDJX Library SDK are those used to interact with the
//! Object Storage (represented by [`storage`] in the EDJX SDK ) and the
//! KV Store (represented by [`kv`] in the EDJX SDK)
//!
//! The Object Store is a decentralized datastore backed by the EDJX P2P network.
//! This SDK has GET and PUT methods to be used for file contents and associated attributes.
//!
//! The KV Store is a decentralized Key-Value datastore backed by the EDJX P2P network.
//! There are specific limits on the size of Key and Value. Key size
//! is limited to 512 bytes and Value is limited to 512 KB.
//!
//!
//! ## Logging
//! This SDK provides generic Rust-compatible logger APIs similar to
//! [`standard RUST Log`](https://docs.rs/log/0.4.14/log/).

pub mod error;
pub mod fetch;
mod fetch_response;
pub mod kv;

#[doc(hidden)]
pub mod logger;

pub mod request;
pub mod response;
mod result;
pub mod storage;
mod storage_response;
mod stream;
mod utils;

pub use ::http::{Method, StatusCode, Uri};
pub use fetch::HttpFetch;
pub use fetch_response::FetchResponse;
pub use http::header::{HeaderMap, HeaderName, HeaderValue};
pub use logger::LogLevel;
pub use request::{HttpMethod, HttpRequest};
pub use response::HttpResponse;
pub use storage_response::{FileAttributes, StorageResponse};
