//! APIs for interacting with EDJX KV-Store.

use crate::error::Error as ErrorCode;
use num_traits::FromPrimitive;
use std::time::Duration;
use thiserror::Error;

#[link(wasm_import_module = "kv")]
extern "C" {
    fn host_kv_get(
        key_ptr: *const u8,
        key_len: u32,
        sd_ptr: *mut u32,
        err_code_ptr: *mut i32,
    ) -> i32;
    fn host_kv_put(
        key_ptr: *const u8,
        key_len: u32,
        val_ptr: *const u8,
        val_len: u32,
        has_ttl: u32,
        ttl: u64,
        error_code_ptr: *mut i32,
    ) -> i32;
    fn host_kv_delete(key_ptr: *const u8, key_len: u32, error_code_ptr: *mut i32) -> i32;
}

const ERROR_UNKNOWN: i32 = 1;
const ERROR_NOT_FOUND: i32 = 2;
const ERROR_UNAUTHORIZED: i32 = 4;

/// Enums describing the errors that correspond to kv modules.
///
/// This is returned by methods similar to
/// [`crate::kv::set()`][`crate::kv::get()`] for
/// error scenarios.
#[derive(Error, Debug)]
pub enum KVError {
    #[error("unknown")]
    /// An unknown error has ocurred while trying to perform a KV store operation.
    Unknown,
    #[error("key not found")]
    /// The value associated with the key could not be found.
    NotFound,
    #[error("Unauthorized")]
    /// This application is not authorized to access this key.
    UnAuthorized,
}

/// Returns the value associated with the provided key.
///
/// # Example
///
/// ```
/// let some_key = "some_key".to_owned();
///
/// let value = match kv::get(&some_key) {
///     Err(e) => {
///         let status = match e {
///             KVError::Unknown => StatusCode::BAD_REQUEST,
///             KVError::UnAuthorized => StatusCode::UNAUTHORIZED,
///             KVError::NotFound  => StatusCode::NOT_FOUND,
///         };
///         return HttpResponse::from(e.to_string()).set_status(status);
///     }
///     Ok(val) => val,
/// };
/// ```
pub fn get<S>(key: S) -> Result<Vec<u8>, KVError>
where
    S: AsRef<str>,
{
    let key_str = key.as_ref();
    let mut sd: u32 = 0;
    let mut err_code = 0;

    let res = unsafe {
        host_kv_get(
            key_str.as_ptr(),
            key_str.len() as u32,
            &mut sd,
            &mut err_code,
        )
    };

    if res < 0 {
        Err(KVError::from(ErrorCode::from_i32(err_code).unwrap()))
    } else {
        let size = unsafe { super::stream::host_stream_size(sd) };

        let mut val = Vec::with_capacity(size as usize);
        unsafe {
            let n = super::stream::host_stream_read_n(sd, val.as_mut_ptr(), size);
            val.set_len(n as usize);
            super::stream::host_stream_drop(sd);
        }

        Ok(val)
    }
}

/// Inserts a key-value pair into the KV store.
///
/// # Example
///
/// ```
/// let some_key = "some_key".to_owned();
/// let some_value = "some_value".to_owned();
///
/// let operation_success : bool = match kv::put(&some_key, some_value, Some(Duration::from_secs(1000 * 5 * 60))) {
///     Err(_) => false,
///     Ok(_) => true,
/// }
///
/// ```
pub fn put<K, V>(key: K, val: V, ttl: Option<Duration>) -> Result<(), KVError>
where
    K: AsRef<str>,
    V: AsRef<[u8]>,
{
    let key_str = key.as_ref();
    let mut err = 0;
    let val_slice = val.as_ref();

    let res = unsafe {
        host_kv_put(
            key_str.as_ptr(),
            key_str.len() as u32,
            val_slice.as_ptr(),
            val_slice.len() as u32,
            ttl.is_some() as u32,
            match ttl {
                Some(d) => d.as_secs(),
                None => 0,
            },
            &mut err,
        )
    };

    if res < 0 {
        Err(KVError::from(ErrorCode::from_i32(err).unwrap()))
    } else {
        Ok(())
    }
}

/// Removes an entry from the KV store.
pub fn delete<K>(key: K) -> Result<(), KVError>
where
    K: AsRef<str>,
{
    let key_str = key.as_ref();
    let mut err = 0;

    let res = unsafe { host_kv_delete(key_str.as_ptr(), key_str.len() as u32, &mut err) };

    if res < 0 {
        Err(KVError::from(ErrorCode::from_i32(err).unwrap()))
    } else {
        Ok(())
    }
}
