//! Error handling enums and utilities.

use http::StatusCode;
use num_derive::FromPrimitive;
use num_traits::FromPrimitive;
use std::convert::TryFrom;
use thiserror::Error;

use crate::kv::KVError;

#[derive(Error, Debug, FromPrimitive)]
pub(crate) enum Error {
    #[error("unknown")]
    Unknown = -1,
    #[error("http: invalid method")]
    HTTPInvalidMethod = 0,
    #[error("http: invalid status code")]
    HTTPInvalidStatusCode = 1,
    #[error("http: invalid version")]
    HTTPInvalidVersion = 2,
    #[error("http: invalid header name")]
    HTTPInvalidHeaderName = 3,
    #[error("http: invalid header value")]
    HTTPInvalidHeaderValue = 4,
    #[error("http: invalid uri")]
    HTTPInvalidUri = 5,
    #[error("http: invalid uri parts")]
    HTTPInvalidUriParts = 6,
    #[error("http: body size exceeds limit")]
    HTTPTooLargeBody = 7,
    #[error("http: URI length exceeds limit")]
    HTTPTooLargeURI = 8,
    #[error("http: header name length exceeds limit")]
    HTTPTooLargeHeaderName = 9,
    #[error("http: header value length exceeds limit")]
    HTTPTooLargeHeaderValue = 10,

    //storage
    #[error("Storage: Empty content ")]
    StorageEmptyContent = 11,
    #[error("Storage: file_name is required")]
    StorageMissingFileName = 12,
    #[error("Storage: bucket_id is deleted")]
    StorageInvalidBucketID = 13,
    #[error("Storage: bucket_id is required")]
    StorageMissingBucketID = 14,
    #[error("Storage: Internal Error")]
    StorageInternalError = 15,
    #[error("Storage: datastore: not found")]
    StorageContentNotFound = 16,
    #[error("Storage: Unauthorized")]
    StorageUnAuthorized = 17,
    #[error("Storage: Content does not exist or deleted")]
    StorageContentDeleted = 18,
    #[error("Storage: Resource limit exceeded")]
    StorageResourceLimit = 21,

    #[error("kv: key not found")]
    KVNotFound = 19,
    #[error("kv: unauthorised operation")]
    KVUnauthorised = 20,




}

/// Enums describing the errors that correspond to HTTP modules.
///
/// This is returned by methods similar to
/// [`crate::fetch::send()`][`crate::request::HttpRequest::from_client()`] for
/// error scenarios.
#[non_exhaustive]
#[derive(Error, Copy, Clone, Debug, PartialEq, Eq)]
pub enum HttpError {
    #[error("unknown")]
    /// An unknown error occured while performing the request.
    Unknown,
    #[error("http header error : {0}")]
    /// A header related error occured.
    Header(#[from] HeaderError),
    #[error("http body too large error")]
    /// HTTP body exceeds prescribed limits.
    HTTPBodyTooLarge,
    #[error("http uri error : {0}")]
    /// A URI related error occured.
    Uri(#[from] UriError),
    #[error("invalid fetch http method")]
    /// An invalid HTTP method was provided.
    HTTPInvalidMethod,
    #[error("invalid response status code")]
    /// An invalid HTTP status code was provided.
    HTTPInvalidStatusCode,
    #[error("invalid http version")]
    /// An invalid HTTP version was provided.
    HTTPInvalidVersion,
}

impl From<Error> for HttpError {
    fn from(e: Error) -> Self {
        match e {
            Error::Unknown => HttpError::Unknown,
            Error::HTTPInvalidMethod => HttpError::HTTPInvalidMethod,
            Error::HTTPInvalidStatusCode => HttpError::HTTPInvalidStatusCode,
            Error::HTTPInvalidVersion => HttpError::HTTPInvalidVersion,
            Error::HTTPInvalidHeaderName => HttpError::Header(HeaderError::InvalidName),
            Error::HTTPInvalidHeaderValue => HttpError::Header(HeaderError::InvalidValue),
            Error::HTTPInvalidUri | Error::HTTPInvalidUriParts => {
                HttpError::Uri(UriError::InvalidUri)
            }
            Error::HTTPTooLargeBody => HttpError::HTTPBodyTooLarge,
            Error::HTTPTooLargeURI => HttpError::Uri(UriError::TooLarge),
            Error::HTTPTooLargeHeaderName => HttpError::Header(HeaderError::TooLargeName),
            Error::HTTPTooLargeHeaderValue => HttpError::Header(HeaderError::TooLargeValue),
            _ => HttpError::Unknown,
        }
    }
}

/// Enums describing the errors that correspond to storage modules.
///
/// This is returned by methods similar to
/// [`crate::storage::put()`][`crate::storage::get()`] 
/// for error scenarios.
#[non_exhaustive]
#[derive(Error, Debug)]
pub enum StorageError {
    #[error("Empty content")]
    /// Content is required.
    EmptyContent,
    #[error("Storage: Missing query param 'file_name'")]
    /// File name is required.
    MissingFileName,
    #[error("bucket_id is deleted")]
    /// The bucket id is deleted.
    DeletedBucketID,
    #[error("Missing query param 'bucket_id'")]
    /// Bucket ID is required.
    MissingBucketID,
    #[error("Internal Error")]
    /// An internal error occured while performing the request.
    InternalError,
    #[error("bucket/content: not found")]
    /// The content or bucket wasn't found on the store.
    ContentNotFound,
    #[error("Storage: Unauthorized")]
    /// The credentials or policies entered are incorrect. Request could not be performed.
    UnAuthorized,
    #[error("Missing Attributes")]
    /// Attributes are required.
    MissingAttributes,
    #[error("Content does not exist or deleted")]
    /// The required content does not exist or has been deleted.
    ContentDeleted,
    #[error("Invalid Attributes")]
    /// The attributes are invalid.
    InvalidAttributes(#[from] serde_json::error::Error),
    #[error("Resource limit exceeded")]
    ResourceLimit,
}

impl From<Error> for StorageError {
    fn from(e: Error) -> Self {
        match e {
            Error::StorageEmptyContent => StorageError::EmptyContent,
            Error::StorageMissingFileName => StorageError::MissingFileName,
            Error::StorageInvalidBucketID => StorageError::DeletedBucketID,
            Error::StorageMissingBucketID => StorageError::MissingBucketID,
            Error::StorageInternalError => StorageError::InternalError,
            Error::StorageContentNotFound => StorageError::ContentNotFound,
            Error::StorageContentDeleted => StorageError::ContentDeleted,
            Error::StorageResourceLimit => StorageError::ResourceLimit,
            Error::StorageUnAuthorized => StorageError::UnAuthorized,
            _ => StorageError::InternalError,
        }
    }
}

impl StorageError {
    pub fn to_http_status_code(&self) -> StatusCode {
        match &self {
            StorageError::EmptyContent
            | StorageError::MissingFileName
            | StorageError::MissingBucketID
            | StorageError::MissingAttributes
            | StorageError::InvalidAttributes(_) => StatusCode::BAD_REQUEST,
            StorageError::ContentNotFound | StorageError::ContentDeleted => StatusCode::NOT_FOUND,
            StorageError::UnAuthorized => StatusCode::FORBIDDEN,
            StorageError::ResourceLimit => StatusCode::UNPROCESSABLE_ENTITY,
            StorageError::DeletedBucketID | StorageError::InternalError => {
                StatusCode::INTERNAL_SERVER_ERROR
            }
        }
    }
}

/// Enums describing the errors that correspond to HTTP headers.
///
/// This is used as a nested error inside
/// [`crate::error::HttpError`]
#[non_exhaustive]
#[derive(Error, Debug, FromPrimitive, Copy, Clone, PartialEq, Eq)]
pub enum HeaderError {
    #[error("invalid header name")]
    /// The header key name is invalid.
    InvalidName = Error::HTTPInvalidHeaderName as isize,
    #[error("invalid header value")]
    /// The header value is invalid.
    InvalidValue = Error::HTTPInvalidHeaderValue as isize,
    #[error("header name length exceeds limit")]
    /// The header name length exceeds the limits.
    TooLargeName = Error::HTTPTooLargeHeaderName as isize,
    #[error("header value length exceeds limit")]
    /// The header value length exceeds the limits.
    TooLargeValue = Error::HTTPTooLargeHeaderValue as isize,
}

impl From<HeaderError> for Error {
    fn from(e: HeaderError) -> Self {
        Error::from_i32(e as i32).unwrap()
    }
}

/// The HTTP body size exceeds the allowable limits.
#[derive(Error, Debug)]
#[error("HTTP body too large")]
pub struct HTTPBodyTooLargeError();

impl From<HTTPBodyTooLargeError> for HttpError {
    fn from(_: HTTPBodyTooLargeError) -> Self {
        return HttpError::HTTPBodyTooLarge;
    }
}

/// Enums describing the errors that correspond to HTTP URI.
///
/// This is used as a nested error inside
/// [`crate::error::HttpError`]
#[non_exhaustive]
#[derive(Error, Copy, Clone, Debug, PartialEq, Eq)]
pub enum UriError {
    #[error("uri exceeds limit")]
    /// The URI exceeds the limits.
    TooLarge,
    #[error("uri value invalid")]
    /// The URI is invalid.
    InvalidUri,
}

impl From<UriError> for Error {
    fn from(e: UriError) -> Self {
        match e {
            UriError::TooLarge => Error::HTTPTooLargeURI,
            UriError::InvalidUri => Error::HTTPInvalidUri,
        }
    }
}

impl TryFrom<i32> for UriError {
    type Error = &'static str;

    fn try_from(value: i32) -> Result<Self, Self::Error> {
        if value == Error::HTTPTooLargeURI as i32 {
            Ok(UriError::TooLarge)
        } else {
            Err("unknown url related error code")
        }
    }
}

impl From<Error> for KVError {
    fn from(e: Error) -> KVError {
        match e {
            Error::KVNotFound => KVError::NotFound,
            Error::KVUnauthorised => KVError::UnAuthorized,
            _ => KVError::Unknown,
        }
    }
}
