//!  APIs for interacting with the EDJX Object Store.
//!
use super::error::{Error, StorageError};
use super::{FileAttributes, StorageResponse};
use num_traits::FromPrimitive;

#[link(wasm_import_module = "storage")]
extern "C" {
    fn host_storage_get(
        bucket_id_ptr: *const u8,
        bucket_id_len: u32,
        file_name_ptr: *const u8,
        file_name_len: u32,
        err_code_ptr: *mut i32,
    ) -> i32;
    fn host_storage_put(
        bucket_id_ptr: *const u8,
        bucket_id_len: u32,
        file_name_ptr: *const u8,
        file_name_len: u32,
        attributes_ptr: *const u8,
        attributes_len: u32,
        content_ptr: *const u8,
        content_len: u32,
        err_code_ptr: *mut i32,
    ) -> i32;
    fn host_storage_delete(
        bucket_id_ptr: *const u8,
        bucket_id_len: u32,
        file_name_ptr: *const u8,
        file_name_len: u32,
        err_code_ptr: *mut i32,
    ) -> i32;
    fn host_storage_get_attributes(
        bucket_id_ptr: *const u8,
        bucket_id_len: u32,
        file_name_ptr: *const u8,
        file_name_len: u32,
        err_code_ptr: *mut i32,
    ) -> i32;
    fn host_storage_set_attributes(
        bucket_id_ptr: *const u8,
        bucket_id_len: u32,
        file_name_ptr: *const u8,
        file_name_len: u32,
        attributes_ptr: *const u8,
        attributes_len: u32,
        err_code_ptr: *mut i32,
    ) -> i32;
}

/// Returns a file from the EDJX Object Store.
///
/// # Example
///
/// ```
///   let bucket_id = "af66ad83-e55b-4a71-a7d5-6ec3199e42e9"
///   let file_name = "example.txt"
///
///   let mut res_bytes: StorageResponse = match storage::get(&bucket_id, &file_name) {
///     Ok(r) => r,
///     Err(e) => {
///         return HttpResponse::from(e.to_string().as_str().to_owned())
///             .set_status(e.to_http_status_code())
///     }
///  };
/// ```
pub fn get<B, FN>(bucket_id: B, file_name: FN) -> Result<StorageResponse, StorageError>
where
    B: AsRef<str>,
    FN: AsRef<str>,
{
    let bucket_id_str = bucket_id.as_ref();
    let file_name_str = file_name.as_ref();

    if file_name_str.trim().is_empty() {
        return Err(StorageError::MissingFileName);
    }

    if bucket_id_str.trim().is_empty() {
        return Err(StorageError::MissingBucketID);
    }

    let mut error_code: i32 = 0;

    let response_value = unsafe {
        host_storage_get(
            bucket_id_str.as_ptr(),
            bucket_id_str.len() as u32,
            file_name_str.as_ptr(),
            file_name_str.len() as u32,
            &mut error_code,
        )
    };

    if response_value < 0 {
        return Err(StorageError::from(Error::from_i32(error_code).unwrap()));
    }

    Ok(StorageResponse::new()?)
}

/// Uploads a file to the EDJX Object Store.
///
///  # Example
///
/// ```
///   let file_name = "example.txt";
///   let bucket_id = "af66ad83-e55b-4a71-a7d5-6ec3199e42e9";
///
///   let buf_data = b"Sample data for upload";
///   let properties = "Cache-Control=true,Type=text";
///
///   let put_res: StorageResponse = match storage::put(&bucket_id, &file_name,  &properties , &buf_data.as_ref()) {
///     Ok(r) => r,
///     Err(e) => {
///         return HttpResponse::from(e.to_string().as_str().to_owned())
///             .set_status(e.to_http_status_code())
///     }
///   };
/// ```
pub fn put<B, FN, A, O>(
    bucket_id: B,
    file_name: FN,
    properties: A,
    contents: O,
) -> Result<StorageResponse, StorageError>
where
    B: AsRef<str>,
    FN: AsRef<str>,
    A: AsRef<str>,
    O: AsRef<[u8]>,
{
    let bucket_id_str = bucket_id.as_ref();
    let file_name_str = file_name.as_ref();
    let content_data = contents.as_ref();
    let attributes_data = properties.as_ref();

    if file_name_str.trim().is_empty() {
        return Err(StorageError::MissingFileName);
    }

    if bucket_id_str.trim().is_empty() {
        return Err(StorageError::MissingBucketID);
    }

    if content_data.len() == 0 {
        return Err(StorageError::EmptyContent);
    }

    let mut error_code: i32 = 0;

    let response_value = unsafe {
        host_storage_put(
            bucket_id_str.as_ptr(),
            bucket_id_str.len() as u32,
            file_name_str.as_ptr(),
            file_name_str.len() as u32,
            attributes_data.as_ptr(),
            attributes_data.len() as u32,
            content_data.as_ptr(),
            content_data.len() as u32,
            &mut error_code,
        )
    };

    if response_value < 0 {
        return Err(StorageError::from(Error::from_i32(error_code).unwrap()));
    }

    Ok(StorageResponse::new()?)
}

/// Deletes the given file from the EDJX Object Store.
///
///  
/// # Example
///
/// ```
///   let bucket_id = "af66ad83-e55b-4a71-a7d5-6ec3199e42e9";
///   let file_name = "example.txt";
///
///   let mut res_bytes: StorageResponse = match storage::delete(&bucket_id, &file_name) {
///     Ok(r) => r,
///     Err(e) => {
///         return HttpResponse::from(e.to_string().as_str().to_owned())
///             .set_status(e.to_http_status_code())
///     }
///  };
/// ```
pub fn delete<B, FN>(bucket_id: B, file_name: FN) -> Result<StorageResponse, StorageError>
where
    B: AsRef<str>,
    FN: AsRef<str>,
{
    let bucket_id_str = bucket_id.as_ref();
    let file_name_str = file_name.as_ref();

    if file_name_str.trim().is_empty() {
        return Err(StorageError::MissingFileName);
    }

    if bucket_id_str.trim().is_empty() {
        return Err(StorageError::MissingBucketID);
    }

    let mut error_code: i32 = 0;

    let response_value = unsafe {
        host_storage_delete(
            bucket_id_str.as_ptr(),
            bucket_id_str.len() as u32,
            file_name_str.as_ptr(),
            file_name_str.len() as u32,
            &mut error_code,
        )
    };

    if response_value < 0 {
        return Err(StorageError::from(Error::from_i32(error_code).unwrap()));
    }

    Ok(StorageResponse::new()?)
}

/// Returns attributes associated with the given file from the EDJX Object Store.
///
/// # Example
///
/// ```
///   let bucket_id = "af66ad83-e55b-4a71-a7d5-6ec3199e42e9";
///   let file_name = "example.txt";
///
///   let mut attributes: FileAttributes = match storage::get_attributes(&bucket_id, &file_name) {
///    Ok(r) => r,
///    Err(e) => {
///        return HttpResponse::from(e.to_string().as_str().to_owned())
///            .set_status(e.to_http_status_code())
///    }
/// };
/// ```
///
pub fn get_attributes<B, FN>(bucket_id: B, file_name: FN) -> Result<FileAttributes, StorageError>
where
    B: AsRef<str>,
    FN: AsRef<str>,
{
    let bucket_id_str = bucket_id.as_ref();
    let file_name_str = file_name.as_ref();

    if file_name_str.trim().is_empty() {
        return Err(StorageError::MissingFileName);
    }

    if bucket_id_str.trim().is_empty() {
        return Err(StorageError::MissingBucketID);
    }

    let mut error_code: i32 = 0;

    let response_value = unsafe {
        host_storage_get_attributes(
            bucket_id_str.as_ptr(),
            bucket_id_str.len() as u32,
            file_name_str.as_ptr(),
            file_name_str.len() as u32,
            &mut error_code,
        )
    };

    if response_value < 0 {
        return Err(StorageError::from(Error::from_i32(error_code).unwrap()));
    }

    let mut resp = StorageResponse::new()?;

    Ok(FileAttributes::from_bytes(resp.body())?)
}

/// Sets the attributes associated with the given file.
///
/// This setting replaces the existing attributes.
///
/// # Example
///
/// ```
///   let bucket_id = "af66ad83-e55b-4a71-a7d5-6ec3199e42e9";
///   let file_name = "example.txt";
///
///   let mut properties: HashMap<String, String> = HashMap::new();
///   properties.insert("Content-Type".to_owned(), "image/jpeg".to_owned());
///   properties.insert("Cache-Control".to_owned(), "image/jpeg".to_owned());
///
///   let new_attributes = FileAttributes::new(Some(properties), None);
///
///   let mut res_bytes: StorageResponse = match storage::set_attributes(&bucket_id, &file_name, new_attributes) {
///    Ok(r) => r,
///    Err(e) => {
///        return HttpResponse::from(e.to_string().as_str().to_owned())
///            .set_status(e.to_http_status_code())
///    }
/// };
/// ```
///
pub fn set_attributes<B, FN>(
    bucket_id: B,
    file_name: FN,
    attributes: FileAttributes,
) -> Result<StorageResponse, StorageError>
where
    B: AsRef<str>,
    FN: AsRef<str>,
{
    let bucket_id_str = bucket_id.as_ref();
    let file_name_str = file_name.as_ref();
    let attributes_data = serde_json::to_vec(&attributes)?;

    if file_name_str.trim().is_empty() {
        return Err(StorageError::MissingFileName);
    }

    if bucket_id_str.trim().is_empty() {
        return Err(StorageError::MissingBucketID);
    }

    if attributes_data.len() == 0 {
        return Err(StorageError::MissingAttributes);
    }

    let mut error_code: i32 = 0;

    let response_value = unsafe {
        host_storage_set_attributes(
            bucket_id_str.as_ptr(),
            bucket_id_str.len() as u32,
            file_name_str.as_ptr(),
            file_name_str.len() as u32,
            attributes_data.as_ptr(),
            attributes_data.len() as u32,
            &mut error_code,
        )
    };

    if response_value < 0 {
        return Err(StorageError::from(Error::from_i32(error_code).unwrap()));
    }

    Ok(StorageResponse::new()?)
}
