//! APIs used to send HTTP Responses for HTTP-triggered serverless functions.

use super::error::{Error, HttpError};
use bytes::Bytes;
use http::header::{HeaderMap, HeaderName, HeaderValue};
use http::{Response, StatusCode, Version};
use num_traits::FromPrimitive;

#[link(wasm_import_module = "http_response")]
extern "C" {
    fn host_response_send(
        status: u32,
        version_addr: *const u8,
        version_len: u32,
        headers_addr: *const u8,
        headers_len: u32,
        body_addr: *const u8,
        body_len: u32,
        err_ptr: *mut i32,
    ) -> i32;
}

/// An HTTP response which may include body, headers, and status code.
///
/// Each serverless function can send only one response to the client.
///
/// ## Sending HTTP Response to client
///
/// - [`HttpResponse::send()`]
/// - [`HttpResponse::send_using_standard_http_lib()`]
///
/// Normally, these methods do not need to be used explicity. EDJX sample
/// [`Rust Serverless Application`](https://github.com/edjx/edjfunction-example-rust)
/// code file (`lib.rs`) uses one of these methods to process Responses and relay to the client.
///
///  ```
/// #[no_mangle]
/// pub fn init() {
///   let req = match HttpRequest::from_client(true);
///
///   let res = crate::serverless_function::serverless(req.unwrap());
///   
///   match res.send() {
///       Ok(x) => x,
///       Err(e) => {
///           error!("{}", e.to_string().as_str());
///       }
///   };
/// }
/// ```
/// Developers need to change only the above `init()` function in `lib.rs` when the HTTP Response type needs to be modified.
///
/// ## Construction of HTTP Responses
///
/// - [`HttpResponse::new()`]
/// - [`HttpResponse::from()`]
///
/// For interoperability with other Rust libraries, [`HttpResponse`] can be initiated using
/// [`http`], which is crate's [`http::Response`] type along with the [`HttpResponse::send_using_standard_http_lib()`]
/// function passing [`http::Response<Option<Bytes>>`] as parameters.
///
/// ## Builder-style Methods
///
/// [`HttpResponse`] can be used as a
/// [builder](https://doc.rust-lang.org/1.0.0/style/ownership/builders.html), allowing responses to
/// be constructed and used through method chaining. Methods with the `set_` name prefix, such as
/// [`set_header()`][`Self::set_text()`], return `Self` to allow chaining. The builder style is
/// typically most useful when constructing and using a response in a single expression. For
/// example:
///
/// ```no_run
/// use edjx::HttpResponse;
/// HttpResponse::new()
///     .set_header("edjx-header".parse().unwrap(), "hello-there".parse().unwrap())
///     .set_status(StatusCode::OK)
///     .send();
/// ```
pub struct HttpResponse {
    parts: Parts,
    body: Option<Vec<u8>>,
}

/// Component parts of `HttpResponse`
///
/// The HTTP response head consists of a status, version, and a set of
/// header fields.
#[doc(hidden)]
struct Parts {
    status: StatusCode,
    version: Version,
    headers: HeaderMap,
}

#[doc(hidden)]
impl Parts {
    /// Creates a new default instance of `Parts` with status code `200 OK`, version as [`HTTP/1.1`] and no headers.
    fn new() -> Parts {
        Parts {
            status: StatusCode::OK,
            version: Version::HTTP_11,
            headers: HeaderMap::new(),
        }
    }
}

impl HttpResponse {
    /// Creates a client [`HttpResponse`].
    ///
    /// The response is created with status code `200 OK`, no headers, and an empty body.
    pub fn new() -> HttpResponse {
        HttpResponse {
            body: None,
            parts: Parts::new(),
        }
    }

    /// Sets the HTTP status code of the client response.
    ///
    /// # Example
    ///
    /// Using the constants from [`http::StatusCode`]:
    ///
    /// ```no_run
    /// use edjx::HttpResponse;
    /// use http::StatusCode;
    ///
    /// let mut resp = HttpResponse::from("not found!").set_status(StatusCode::NOT_FOUND);
    /// resp.send();
    /// ```
    pub fn set_status(mut self, status: StatusCode) -> Self {
        self.parts.status = status;
        return self;
    }

    /// Returns the HTTP status code of the response.
    pub fn get_status(&self) -> StatusCode {
        self.parts.status
    }

    /// Sets the HTTP version of this response. Use constants from [`http::Version`]:
    pub fn set_version(mut self, version: Version) -> Self {
        self.parts.version = version;
        return self;
    }

    /// Returns the HTTP version of this response.
    pub fn get_version(&self) -> Version {
        return self.parts.version;
    }

    /// Sets the given [`&str`] value as the body of the client response.
    ///
    /// Any previous body that was set on the response is discarded.
    pub fn set_text(mut self, text: &str) -> Self {
        self.body = Some(Vec::from(text));
        return self;
    }

    /// Sets the given [`Vec<u8>`] value as the body of the client response.
    ///
    /// Any previous body that was set on the response is discarded.
    pub fn set_vec_bytes(mut self, bytes: Vec<u8>) -> Self {
        self.body = Some(bytes);
        return self;
    }

    /// Sets a response header to the given value, discarding any previous values for the given
    /// header name.
    ///
    /// # Example
    ///
    /// ```no_run
    /// use edjx::HttpResponse;
    ///
    /// let mut resp = HttpResponse::new().set_header("hello".parse().unwrap(), "world!".parse().unwrap());
    ///
    /// let header_map = resp.get_headers();
    /// assert!(header_map.contains_key("hello"));
    /// ```
    pub fn set_header(mut self, header_name: HeaderName, value: HeaderValue) -> Self {
        self.parts.headers.append(header_name, value);
        return self;
    }

    /// Returns the response header map as [`http::header::HeaderMap`].
    pub fn get_headers(&self) -> &HeaderMap {
        return &self.parts.headers;
    }

    /// Returns the response header map as a mutable reference to [`http::header::HeaderMap`].
    pub fn headers_mut(&mut self) -> &mut HeaderMap {
        return &mut self.parts.headers;
    }

    /// Send the response to the client.
    ///
    /// # Error Response
    ///
    /// This method returns an error response of [`HttpError::Unknown`] if another response was sent to the client
    /// by the method [`HttpResponse::send_using_standard_http_lib()`], or even if http connection from client has been closed.
    ///
    /// # Example
    /// ```
    /// #[no_mangle]
    /// pub fn init() {
    ///   let req = match HttpRequest::from_client(true)
    ///
    ///   let res = crate::serverless_function::serverless(req);
    ///     match res.send() {
    ///         Ok(x) => x,
    ///         Err(e) => {
    ///             error!("{}", e.to_string().as_str());
    ///         }
    ///     };
    /// }
    /// ```
    pub fn send(&self) -> Result<(), HttpError> {
        let headers_vec =
            serde_json::to_vec(&super::utils::convert_to_hashmap(&self.parts.headers)).unwrap();
        let headers_slice = headers_vec.as_slice();
        let status = self.parts.status.as_u16();
        let version: String = super::utils::to_version_string(&self.parts.version);
        let body_len: u32;
        let body_slice;
        let empty_body_vec = Vec::new();
        if self.body.is_none() {
            body_len = 0;
            body_slice = empty_body_vec.as_slice();
        } else {
            body_slice = self.body.as_ref().unwrap().as_slice();
            body_len = body_slice.len() as u32;
        }

        let mut err = 0;
        let resp = unsafe {
            host_response_send(
                status as u32,
                version.as_ptr(),
                version.len() as u32,
                headers_slice.as_ptr(),
                headers_slice.len() as u32,
                body_slice.as_ptr(),
                body_len,
                &mut err,
            )
        };

        if resp < 0 {
            return Err(HttpError::from(Error::from_i32(err).unwrap()));
        }

        Ok(())
    }

    /// Send the response to the client using [`http::Response`].
    ///
    ///
    /// # Error Response
    ///
    /// This method returns an error response of [`HttpError::Unknown`] if another response was sent.
    ///
    /// # Example
    ///
    /// Sending a response using [`http::Response`]:
    ///
    /// ```no_run
    /// use edjx::HttpResponse;
    /// use http::{Response, StatusCode};
    ///
    /// let mut response = Response::builder();
    /// response.header("Foo", "Bar").status(StatusCode::OK);
    /// HttpResponse::send_using_standard_http_lib(response.body(()));
    ///
    /// ```
    pub fn send_using_standard_http_lib(res: Response<Option<Bytes>>) -> Result<(), HttpError> {
        let headers_vec =
            serde_json::to_vec(&super::utils::convert_to_hashmap(&res.headers())).unwrap();
        let headers_slice = headers_vec.as_slice();
        let status = res.status().as_u16();
        let version: String = super::utils::to_version_string(&res.version());
        let body = match res.body() {
            None => Default::default(),
            Some(body) => body.as_ref(),
        };
        let mut err = 0;
        let resp = unsafe {
            host_response_send(
                status as u32,
                version.as_ptr(),
                version.len() as u32,
                headers_slice.as_ptr(),
                headers_slice.len() as u32,
                body.as_ptr(),
                body.len() as u32,
                &mut err,
            )
        };

        if resp < 0 {
            return Err(HttpError::from(Error::from_i32(err).unwrap()));
        }

        Ok(())
    }
}

impl From<String> for HttpResponse {
    #[inline]
    fn from(s: String) -> HttpResponse {
        HttpResponse {
            body: Some(Vec::from(s)),
            parts: Parts::new(),
        }
    }
}

impl From<&str> for HttpResponse {
    #[inline]
    fn from(s: &str) -> HttpResponse {
        HttpResponse {
            body: Some(Vec::from(s)),
            parts: Parts::new(),
        }
    }
}

impl From<Vec<u8>> for HttpResponse {
    #[inline]
    fn from(v: Vec<u8>) -> HttpResponse {
        HttpResponse {
            body: Some(v),
            parts: Parts::new(),
        }
    }
}
