/// Logs a message at the error level.
///
/// # Example
///
/// ```
/// use edjx::error;
///
/// # fn init() {
/// let (err_info, port) = ("No connection", 22);
///
/// error!("Error: {} on port {}", err_info, port);
/// error!(target: "app_events", "App Error: {}, Port: {}", err_info, 22);
/// # }
/// ```
#[macro_export(local_inner_macros)]
macro_rules! error {
    ($($arg:tt)+) => (
        $crate::logger::call_host_logger(
            __log_format_args!($($arg)+),
            $crate::LogLevel::ERROR,
        );
    )
}

/// Logs a message at the info level.
///
/// # Example
///
/// ```
/// use edjx::info;
///
/// # fn main() {
/// # struct Connection { port: u32, speed: f32 }
/// let conn_info = Connection { port: 40, speed: 3.20 };
///
/// info!("Connected to port {} at {} Mb/s", conn_info.port, conn_info.speed);
/// info!(target: "connection_events", "Successfull connection, port: {}, speed: {}",
///       conn_info.port, conn_info.speed);
/// # }
/// ```
#[macro_export(local_inner_macros)]
macro_rules! info {
    ($($arg:tt)+) => (
        $crate::logger::call_host_logger(
            __log_format_args!($($arg)+),
            $crate::LogLevel::INFO,
        );
    )
}

/// Logs a message at the warn level.
///
/// # Example
///
/// ```
/// use edjx::warn;
///
/// # fn main() {
/// let warn_description = "Invalid Input";
///
/// warn!("Warning! {}!", warn_description);
/// warn!(target: "input_events", "App received warning: {}", warn_description);
/// # }
/// ```
#[macro_export(local_inner_macros)]
macro_rules! warn {
    ($($arg:tt)+) => (
        $crate::logger::call_host_logger(
            __log_format_args!($($arg)+),
            $crate::LogLevel::WARN,
        );
    )
}

/// Logs a message at the debug level.
///
/// # Example
///
/// ```
/// use edjx::debug;
///
/// # fn main() {
/// # struct Position { x: f32, y: f32 }
/// let pos = Position { x: 3.234, y: -1.223 };
///
/// debug!("New position: x: {}, y: {}", pos.x, pos.y);
/// debug!(target: "app_events", "New position: x: {}, y: {}", pos.x, pos.y);
/// # }
/// ```
#[macro_export(local_inner_macros)]
macro_rules! debug {
    ($($arg:tt)+) => (
        $crate::logger::call_host_logger(
            __log_format_args!($($arg)+),
            $crate::LogLevel::DEBUG,
        );
    )
}

/// Logs a message at the trace level.
///
/// # Example
///
/// ```
/// use edjx::trace;
///
/// # fn main() {
/// # struct Position { x: f32, y: f32 }
/// let pos = Position { x: 3.234, y: -1.223 };
///
/// trace!("Position is: x: {}, y: {}", pos.x, pos.y);
/// trace!(target: "app_events", "x is {} and y is {}",
///        if pos.x >= 0.0 { "positive" } else { "negative" },
///        if pos.y >= 0.0 { "positive" } else { "negative" });
/// # }
/// ```
#[macro_export(local_inner_macros)]
macro_rules! trace {
    ($($arg:tt)+) => (
        $crate::logger::call_host_logger(
            __log_format_args!($($arg)+),
            $crate::LogLevel::TRACE,
        );
    )
}

/// Logs a message at the fatal level.
///
/// # Example
///
/// ```
/// use edjx::fatal;
///
/// # fn init() {
/// let (fatal_info, port) = ("Failed connection", 22);
///
/// error!("Fatal: {} on port {}", fatal_info, port);
/// error!(target: "app_events", "App Error: {}, Port: {}", fatal_info, port);
/// # }
/// ```
#[macro_export(local_inner_macros)]
macro_rules! fatal {
    ($($arg:tt)+) => (
        $crate::logger::call_host_logger(
            __log_format_args!($($arg)+),
            $crate::LogLevel::FATAL,
        );
    )
}

#[doc(hidden)]
#[macro_export]
macro_rules! __log_format_args {
    ($($args:tt)*) => {
        format_args!($($args)*)
    };
}
