use super::error::{Error, HTTPBodyTooLargeError, HeaderError, HttpError};
use http::{header::HeaderName, HeaderMap, StatusCode};
use num_traits::FromPrimitive;
use std::str::{from_utf8, FromStr};

#[link(wasm_import_module = "fetch_response")]
extern "C" {
    fn http_fetch_response_status_code() -> i32;
    fn http_fetch_response_get_body(err_ptr: *mut i32) -> i32;
    fn http_fetch_response_get_all_header_keys(error_code_ptr: *mut i32) -> i32;
    fn http_fetch_response_get_header_value(
        header_ptr: *const u8,
        header_len: i32,
        err_code_ptr: *mut i32,
    ) -> i32;
}

/// Response for HTTP fetch request, which may include body, headers, and status code.
///
/// ```no_run
/// use edjx::{FetchResponse, HttpFetch};
/// use http::StatusCode;
///
/// let fetch_uri = Uri::from_str("https://httpbin.org/get").unwrap();
/// let mut fetch_res: FetchResponse = match HttpFetch::get(fetch_uri).send() {
///    Ok(resp) => resp,
///    Err(e) => {
///        panic!("{}", &e.to_string());
///    }
/// };
///
/// let body = fetch_res.body();
/// assert_eq!(fetch_res.status_code(), &StatusCode::OK);
/// ```
#[derive(Debug)]
pub struct FetchResponse {
    status: StatusCode,
    headers: HeaderMap,
    body: Option<Vec<u8>>,
}

impl FetchResponse {
    #[doc(hidden)]
    pub(crate) fn from_server() -> Result<FetchResponse, HttpError> {
        let status = get_status_code_from_host();

        let mut headers = HeaderMap::new();
        let header_keys = get_all_header_keys_from_host()?;
        for key in header_keys {
            headers.insert(
                HeaderName::from_str(key.as_str()).unwrap(),
                get_header_value_from_host(key.as_str())?.parse().unwrap(),
            );
        }

        let body = get_body_from_host()?;
        Ok(FetchResponse {
            status,
            headers,
            body: Some(body),
        })
    }

    /// Returns the HTTP status code of the fetch response.
    pub fn status_code(&self) -> &StatusCode {
        return &self.status;
    }

    /// Returns the HTTP header map of the fetch response.
    pub fn headers(&self) -> &HeaderMap {
        return &self.headers;
    }

    /// Returns the HTTP body of the fetch response.
    pub fn body(&mut self) -> Vec<u8> {
        match &self.body {
            Some(body_vec) => body_vec.clone(),
            None => Vec::new(),
        }
    }
}

fn get_status_code_from_host() -> StatusCode {
    let response = unsafe { http_fetch_response_status_code() };

    StatusCode::from_u16(response as u16).unwrap()
}

fn get_body_from_host() -> Result<Vec<u8>, HTTPBodyTooLargeError> {
    let mut err: i32 = 0;
    let response = unsafe { http_fetch_response_get_body(&mut err) };

    if response > 0 {
        Ok(super::result::get_result_bytes(response).unwrap())
    } else if response == 0 {
        Ok(Vec::with_capacity(0))
    } else {
        assert!(err == Error::HTTPTooLargeBody as i32);
        return Err(HTTPBodyTooLargeError());
    }
}

fn get_all_header_keys_from_host() -> Result<Vec<String>, HeaderError> {
    let mut err_code: i32 = 0;
    let response = unsafe { http_fetch_response_get_all_header_keys(&mut err_code) };

    if response > 0 {
        let data = super::result::get_result_bytes(response).unwrap();

        let keys_delimited_string = from_utf8(&data).unwrap();
        let mut v = Vec::new();
        for header_key in keys_delimited_string.split(",") {
            if header_key.len() > 0 {
                v.push(String::from(header_key))
            }
        }
        Ok(v)
    } else if response == 0 {
        Ok(Vec::with_capacity(0))
    } else {
        Err(HeaderError::from_i32(err_code).unwrap())
    }
}

fn get_header_value_from_host(header_key: &str) -> Result<String, HeaderError> {
    let mut key = String::from(header_key);
    let key = key.as_mut_str();
    let dest_length = key.len() as i32;
    let dest_pointer = key.as_mut_ptr() as *const u8;
    let mut err: i32 = 0;

    let result_size =
        unsafe { http_fetch_response_get_header_value(dest_pointer, dest_length, &mut err) };
    if result_size > 0 {
        let data = super::result::get_result_bytes(result_size);
        Ok(String::from_utf8(data.unwrap()).unwrap())
    } else if result_size == 0 {
        return Ok(String::with_capacity(0));
    } else {
        return Err(HeaderError::from_i32(err).unwrap());
    }
}
